# -*- coding: utf-8 -*-

import logging
from odoo import _
from odoo.http import request, route
from ..base_api_controller import BaseApiController

_logger = logging.getLogger(__name__)


class AuthApiController(BaseApiController):
    """用户认证相关API控制器"""
    
    @route('/api/apollo/jwt/verify', type='json', auth='public', methods=['POST'], csrf=False)
    def verify_jwt_token(self):
        """验证JWT令牌"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['token'])
            if error:
                return error
            
            token = data['token']
            
            # 验证JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            verify_result = jwt_service.verify_token(token)
            
            if not verify_result['valid']:
                return self._error_response('令牌无效或已过期', 401)
            
            # 构建响应数据
            response_data = {
                'valid': True,
                'phone': verify_result['phone'],
                'status': verify_result['status'],
                'status_description': verify_result['status_description'],
                'expires_at': verify_result['expires_at']
            }
            
            # 如果是已登录状态，添加用户信息
            if verify_result['status'] == 'login_success' and verify_result.get('user_id'):
                user = request.env['res.users'].sudo().browse(verify_result['user_id'])
                if user.exists():
                    response_data.update({
                        'user_id': user.id,
                        'user_name': user.name
                    })
            
            return self._success_response('令牌验证成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "验证JWT令牌")
    
    @route('/api/apollo/user/register', type='json', auth='public', methods=['POST'], csrf=False)
    def register_user(self):
        """完成用户注册"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['token', 'name', 'password']
            )
            if error:
                return error
            
            token = data['token']
            name = data['name']
            password = data['password']
            
            # 验证JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            verify_result = jwt_service.verify_token(token)
            
            if not verify_result['valid']:
                return self._error_response('令牌无效或已过期', 401)
            
            # 检查令牌状态是否为待注册
            if verify_result['status'] != 'registration_pending':
                return self._error_response('令牌状态不正确，无法注册', 400)
            
            phone = verify_result['phone']
            organization_id = verify_result.get('organization_id')
            invitation_code = verify_result.get('invitation_code')
            
            # 确保token中包含organization_id
            if not organization_id:
                return self._error_response('令牌中缺少组织信息', 400)
            
            # 验证组织是否存在
            organization = request.env['apollo.organization'].sudo().search([
                ('id', '=', organization_id),
                ('active', '=', True)
            ], limit=1)
            
            if not organization:
                return self._error_response('组织不存在或已停用', 404)
            
            # 检查用户是否已存在
            existing_user = request.env['res.users'].sudo().search([
                ('mobile', '=', phone)
            ], limit=1)
            
            if existing_user:
                return self._error_response('该手机号已注册', 400)
            
            # 创建用户
            user_data = {
                'name': name,
                'login': phone,  # 使用手机号作为登录名
                'mobile': phone,
                'password': password,  # 设置用户密码
                'active': True,
                'groups_id': [(6, 0, [request.env.ref('base.group_user').id])]
            }
            
            user = request.env['res.users'].sudo().create(user_data)
            
            # 查找并绑定邀请码
            search_domain = [
                ('phone', '=', phone),
                ('organization_id', '=', organization_id),
                ('active', '=', True),
                ('voided', '=', False),
                ('bound_user_id', '=', False)
            ]
            
            # 如果令牌中包含邀请码，优先使用该邀请码
            if invitation_code:
                search_domain.append(('code', '=', invitation_code))
            
            invitation = request.env['apollo.invitation.code'].sudo().search(search_domain, limit=1)
            
            if invitation:
                # 绑定邀请码到用户（这会自动创建组织成员记录并设置owner）
                invitation.bind_user(user.id)
                _logger.info(f"用户注册成功并绑定邀请码: {user.name} -> {invitation.code}")
            else:
                # 如果没有找到邀请码，直接创建组织成员记录
                request.env['apollo.organization.member'].sudo().create({
                    'user_id': user.id,
                    'organization_id': organization_id,
                    'join_date': request.env.cr.now()
                })
                _logger.warning(f"用户注册时未找到对应的邀请码: {phone}, 组织: {organization_id}")
            
            # 生成新的登录令牌
            login_token_data = jwt_service.generate_login_token(
                user_id=user.id,
                phone=phone
            )
            
            # 构建响应数据
            response_data = {
                'phone': phone,
                'status': 'login_success',
                'status_description': '注册成功，已登录',
                'token': login_token_data['token'],
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile
                }
            }
            
            return self._success_response('注册成功', response_data, 201)
            
        except Exception as e:
            return self._handle_exception(e, "用户注册")
    
    @route('/api/apollo/user/login', type='json', auth='public', methods=['POST'], csrf=False)
    def login_user(self):
        """通过手机号或邮箱和密码登录"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['password'],
                optional_fields=['phone', 'email']
            )
            if error:
                return error
            
            phone = data.get('phone')
            email = data.get('email')
            password = data['password']
            
            # 确保提供了手机号或邮箱中的至少一个
            if not phone and not email:
                return self._error_response('请提供手机号或邮箱', 400)
            
            # 查找用户
            search_domain = [('active', '=', True)]
            

            if phone:
                # 只提供了手机号
                search_domain.append(('mobile', '=', phone))
            else:
                # 只提供了邮箱
                search_domain.append(('email', '=', email))
            
            user = request.env['res.users'].sudo().search(search_domain, limit=1)
            
            if not user:
                return self._error_response('用户不存在或已停用', 404)
            
            # 验证密码
            try:
                # 使用 Odoo 的认证机制验证密码
                credential = {
                    'login': user.login,
                    'password': password,
                    'type': 'password',
                }
                auth_result = request.env['res.users'].sudo().authenticate(
                    request.env.cr.dbname, 
                    credential, 
                    {'interactive': False}
                )
                if not auth_result or auth_result.get('uid') != user.id:
                    return self._error_response('密码错误', 401)
            except Exception:
                return self._error_response('密码错误', 401)
            
            # 生成登录令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            login_token_data = jwt_service.generate_login_token(
                user_id=user.id,
                phone=user.mobile  # 使用用户实际的手机号
            )
            
            # 获取用户的组织信息
            member_records = request.env['apollo.organization.member'].sudo().search([
                ('user_id', '=', user.id),
                ('leave_date', '=', False)  # 只获取当前成员身份
            ])
            
            organizations = []
            for member in member_records:
                organizations.append({
                    'id': member.organization_id.id,
                    'name': member.organization_id.name,
                    'join_date': member.join_date.isoformat() if member.join_date else None
                })
            
            # 构建响应数据
            response_data = {
                'phone': user.mobile,  # 使用用户实际的手机号
                'status': 'login_success',
                'status_description': '登录成功',
                'token': login_token_data['token'],
                'expires_at': login_token_data.get('expires_at'),
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile
                },
                'organizations': organizations
            }
            
            return self._success_response('登录成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "用户登录")
    
    @route('/api/apollo/user/login/sms', type='json', auth='public', methods=['POST'], csrf=False)
    def login_with_sms_code(self):
        """通过手机号和短信验证码登录"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['phone', 'sms_code']
            )
            if error:
                return error
            
            phone = data['phone']
            sms_verification_code = data['sms_code']
            
            # 验证短信验证码
            sms_model = request.env['apollo.sms.verification.code'].sudo()
            is_valid, message = sms_model.verify_login_code(phone, sms_verification_code)
            
            if not is_valid:
                return self._error_response(message, 400)
            
            # 查找用户
            user = request.env['res.users'].sudo().search([
                ('mobile', '=', phone),
                ('active', '=', True)
            ], limit=1)
            
            if not user:
                return self._error_response('用户不存在或已停用', 404)
            
            # 生成登录令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            login_token_data = jwt_service.generate_login_token(
                user_id=user.id,
                phone=phone
            )
            
            # 获取用户的组织信息
            member_records = request.env['apollo.organization.member'].sudo().search([
                ('user_id', '=', user.id),
                ('leave_date', '=', False)  # 只获取当前成员身份
            ])
            
            organizations = []
            for member in member_records:
                organizations.append({
                    'id': member.organization_id.id,
                    'name': member.organization_id.name,
                    'join_date': member.join_date.isoformat() if member.join_date else None
                })
            
            # 构建响应数据
            response_data = {
                'phone': phone,
                'status': 'login_success',
                'status_description': '登录成功',
                'token': login_token_data['token'],
                'expires_at': login_token_data.get('expires_at'),
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile
                },
                'organizations': organizations
            }
            
            return self._success_response('登录成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "短信验证码登录")
    
    @route('/api/apollo/user/profile', type='json', auth='public', methods=['POST'], csrf=False)
    def get_user_profile(self):
        """获取用户资料"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['token'])
            if error:
                return error
            
            token = data['token']
            
            # 验证JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            verify_result = jwt_service.verify_token(token)
            
            if not verify_result['valid']:
                return self._error_response('令牌无效或已过期', 401)
            
            # 检查是否为已登录状态
            if verify_result['status'] != 'login_success':
                return self._error_response('用户未登录', 401)
            
            user_id = verify_result.get('user_id')
            if not user_id:
                return self._error_response('令牌中缺少用户信息', 400)
            
            # 获取用户信息
            user = request.env['res.users'].sudo().browse(user_id)
            if not user.exists():
                return self._error_response('用户不存在', 404)
            
            # 获取用户的组织成员信息
            member_records = request.env['apollo.organization.member'].sudo().search([
                ('user_id', '=', user.id),
                ('leave_date', '=', False)  # 只获取当前成员身份
            ])
            
            organizations = []
            for member in member_records:
                organizations.append({
                    'id': member.organization_id.id,
                    'name': member.organization_id.name,
                    'join_date': member.join_date.isoformat() if member.join_date else None
                })
            
            # 构建响应数据
            response_data = {
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile or '',
                    'active': user.active
                },
                'organizations': organizations
            }
            
            return self._success_response('获取用户资料成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "获取用户资料")
