from odoo import api, fields, models, _
from odoo.exceptions import ValidationError
from datetime import datetime


class ApolloOrganizationMember(models.Model):
    _name = 'apollo.organization.member'
    _description = 'Apollo Organization Member'
    _order = 'join_date desc'
    _rec_name = 'display_name'
    
    # SQL 约束优化
    _sql_constraints = [
        ('unique_active_member', 
         'EXCLUDE (organization_id WITH =, user_id WITH =) WHERE (leave_date IS NULL)',
         '用户在同一组织中只能有一个活跃成员记录'),
        ('check_dates_order',
         'CHECK (leave_date IS NULL OR leave_date > join_date)',
         '离开时间必须晚于加入时间'),
    ]

    organization_id = fields.Many2one(
        'apollo.organization', 
        string='组织', 
        required=True, 
        ondelete='cascade',
        index=True
    )
    user_id = fields.Many2one(
        'res.users', 
        string='用户', 
        required=True,
        index=True
    )
    join_date = fields.Datetime(
        string='加入时间', 
        required=True, 
        default=fields.Datetime.now,
        help='用户加入组织的时间'
    )
    leave_date = fields.Datetime(
        string='离开时间',
        help='用户离开组织的时间，为空表示仍在组织中'
    )
    is_active = fields.Boolean(
        string='是否在组织中',
        compute='_compute_is_active',
        store=True,
        help='用户是否仍在组织中'
    )
    is_owner = fields.Boolean(
        string='是否为所有者',
        compute='_compute_is_owner',
        help='该成员是否为组织所有者'
    )
    added_by_id = fields.Many2one(
        'res.users', 
        string='添加人', 
        default=lambda self: self.env.user,
        help='将用户添加到组织的操作人'
    )
    removed_by_id = fields.Many2one(
        'res.users', 
        string='移除人',
        help='将用户从组织中移除的操作人'
    )
    note = fields.Text(
        string='备注',
        help='关于用户加入或离开组织的备注信息'
    )
    invitation_code_id = fields.Many2one(
        'apollo.invitation.code',
        string='邀请码',
        help='用户通过此邀请码加入组织'
    )
    
    # 显示名称
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )

    @api.depends('leave_date')
    def _compute_is_active(self):
        """计算用户是否仍在组织中"""
        for record in self:
            record.is_active = not record.leave_date

    @api.depends('organization_id.owner_id', 'user_id')
    def _compute_is_owner(self):
        """计算该成员是否为组织所有者"""
        for record in self:
            record.is_owner = (record.organization_id.owner_id and 
                              record.user_id.id == record.organization_id.owner_id.id)

    @api.depends('organization_id.name', 'user_id.name', 'join_date', 'leave_date')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.organization_id and record.user_id:
                status = '在职' if record.is_active else '已离职'
                record.display_name = f"{record.organization_id.name} - {record.user_id.name} ({status})"
            else:
                record.display_name = '组织成员记录'

    def name_get(self):
        """自定义名称显示"""
        result = []
        for record in self:
            result.append((record.id, record.display_name))
        return result

    @api.model
    def add_member(self, organization_id, user_id, note=None):
        """添加成员到组织"""
        # 检查用户是否已经在组织中
        existing_active = self.search([
            ('organization_id', '=', organization_id),
            ('user_id', '=', user_id),
            ('leave_date', '=', False)
        ])
        
        if existing_active:
            return existing_active
        
        # 创建新的成员记录
        return self.create({
            'organization_id': organization_id,
            'user_id': user_id,
            'note': note,
        })

    def remove_member(self, note=None):
        """从组织中移除成员"""
        self.ensure_one()
        
        if self.leave_date:
            return False  # 已经离开了
        
        # 检查是否为组织所有者
        if self.organization_id.owner_id and self.user_id.id == self.organization_id.owner_id.id:
            raise ValidationError(_('不能移除组织所有者，请先转移所有权'))
        
        self.write({
            'leave_date': fields.Datetime.now(),
            'removed_by_id': self.env.user.id,
            'note': (self.note + '\n' + note) if self.note and note else (note or self.note)
        })
        
        return True

    @api.model
    def get_active_members(self, organization_id):
        """获取组织的当前成员"""
        return self.search([
            ('organization_id', '=', organization_id),
            ('leave_date', '=', False)
        ])

    @api.model
    def get_member_history(self, organization_id, user_id):
        """获取用户在组织中的历史记录"""
        return self.search([
            ('organization_id', '=', organization_id),
            ('user_id', '=', user_id)
        ], order='join_date desc')

    @api.constrains('join_date', 'leave_date')
    def _check_dates(self):
        """检查日期的有效性"""
        for record in self:
            if record.leave_date and record.join_date and record.leave_date <= record.join_date:
                raise models.ValidationError(_('离开时间必须晚于加入时间'))
    
    def check_user_access(self, access_type='read', user=None):
        """检查用户对成员记录的访问权限
        
        Args:
            access_type: 'read', 'write', 'create', 'unlink'
            user: 用户记录，默认为当前用户
        
        Returns:
            bool: 是否有权限
        """
        self.ensure_one()
        if not user:
            user = self.env.user
        
        # 超级用户有所有权限
        if user._is_superuser():
            return True
        
        # 业务管理员有所有权限
        if self.organization_id.is_user_business_admin(user):
            return True
        
        # 组织所有者有所有权限
        if self.organization_id.is_user_owner(user):
            return True
        
        # 组织成员只有读权限
        if access_type == 'read' and self.organization_id.is_user_member(user):
            return True
        
        return False
    
    @api.model_create_multi
    def create(self, vals_list):
        """创建成员记录时自动分配用户组"""
        records = super(ApolloOrganizationMember, self).create(vals_list)
        
        for record in records:
            if record.is_active:
                record.organization_id.assign_user_groups()
        
        return records
    
    def write(self, vals):
        """更新成员记录时管理用户组"""
        result = super(ApolloOrganizationMember, self).write(vals)
        
        # 如果离开时间发生变更，重新分配用户组
        if 'leave_date' in vals:
            for record in self:
                if vals['leave_date']:  # 用户离开组织
                    record.organization_id.remove_user_groups(record.user_id)
                else:  # 用户重新加入组织
                    record.organization_id.assign_user_groups()
        
        return result