from odoo import api, fields, models, _
from datetime import datetime


class ApolloOrganizationOwnerChange(models.Model):
    _name = 'apollo.organization.owner.change'
    _description = 'Apollo Organization Owner Change Log'
    _order = 'change_date desc'
    _rec_name = 'display_name'

    organization_id = fields.Many2one(
        'apollo.organization', 
        string='组织', 
        required=True, 
        ondelete='cascade'
    )
    old_owner_id = fields.Many2one(
        'res.users', 
        string='原所有者',
        help='变更前的组织所有者'
    )
    new_owner_id = fields.Many2one(
        'res.users', 
        string='新所有者', 
        required=True,
        help='变更后的组织所有者'
    )
    changed_by_id = fields.Many2one(
        'res.users', 
        string='操作人', 
        required=True, 
        default=lambda self: self.env.user,
        help='执行变更操作的用户'
    )
    change_date = fields.Datetime(
        string='变更时间', 
        required=True, 
        default=fields.Datetime.now,
        help='所有者变更的时间'
    )
    reason = fields.Text(
        string='变更原因',
        help='记录变更所有者的原因'
    )
    
    # 显示名称
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )

    @api.depends('organization_id.name', 'old_owner_id.name', 'new_owner_id.name', 'change_date')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.organization_id and record.new_owner_id:
                old_owner_name = record.old_owner_id.name if record.old_owner_id else '无'
                record.display_name = f"{record.organization_id.name}: {old_owner_name} → {record.new_owner_id.name}"
            else:
                record.display_name = '组织所有者变更记录'

    def name_get(self):
        """自定义名称显示"""
        result = []
        for record in self:
            result.append((record.id, record.display_name))
        return result
