# -*- coding: utf-8 -*-

import jwt
import logging
from datetime import datetime, timedelta
from odoo import api, models, _
from odoo.exceptions import ValidationError

_logger = logging.getLogger(__name__)


class JwtService(models.AbstractModel):
    """JWT令牌服务"""
    
    _name = 'apollo.jwt.service'
    _description = 'JWT Token Service'

    # 状态常量
    STATUS_LOGIN_SUCCESS = 'login_success'
    STATUS_REGISTRATION_PENDING = 'registration_pending'
    
    # 状态描述映射
    STATUS_DESCRIPTIONS = {
        STATUS_LOGIN_SUCCESS: '登录成功',
        STATUS_REGISTRATION_PENDING: '待注册'
    }

    @api.model
    def get_jwt_secret(self):
        """获取JWT密钥
        
        Returns:
            str: JWT密钥
        """
        return self.env['ir.config_parameter'].sudo().get_param(
            'apollo.jwt.secret', 
            'apollo_default_secret_key_change_in_production'
        )

    @api.model
    def generate_login_token(self, user_id, phone, expires_hours=24):
        """生成登录成功的JWT令牌
        
        Args:
            user_id (int): 用户ID
            phone (str): 手机号
            expires_hours (int): 过期小时数
            
        Returns:
            dict: 包含token和相关信息的字典
        """
        try:
            user = self.env['res.users'].sudo().browse(user_id)
            if not user.exists():
                raise ValidationError(_('用户不存在'))
            
            user_data = {
                'id': user.id,
                'name': user.name,
                'email': user.email or '',
                'mobile': user.mobile or phone
            }
            
            token = self._generate_token(
                phone=phone,
                status=self.STATUS_LOGIN_SUCCESS,
                user_data=user_data,
                expires_hours=expires_hours
            )
            
            return {
                'token': token,
                'status': self.STATUS_LOGIN_SUCCESS,
                'status_description': self.STATUS_DESCRIPTIONS[self.STATUS_LOGIN_SUCCESS],
                'user_data': user_data,
                'expires_at': int((datetime.utcnow() + timedelta(hours=expires_hours)).timestamp())
            }
            
        except Exception as e:
            _logger.error(f"生成登录令牌失败: {str(e)}")
            raise ValidationError(_('生成登录令牌失败: %s') % str(e))

    @api.model
    def generate_registration_token(self, phone, organization_id=None, invitation_code=None, expires_hours=1):
        """生成注册待完成的JWT令牌
        
        Args:
            phone (str): 手机号
            organization_id (int): 组织ID
            invitation_code (str): 邀请码
            expires_hours (int): 过期小时数（注册令牌通常较短）
            
        Returns:
            dict: 包含token和相关信息的字典
        """
        try:
            token = self._generate_token(
                phone=phone,
                status=self.STATUS_REGISTRATION_PENDING,
                organization_id=organization_id,
                invitation_code=invitation_code,
                expires_hours=expires_hours
            )
            
            return {
                'token': token,
                'status': self.STATUS_REGISTRATION_PENDING,
                'status_description': self.STATUS_DESCRIPTIONS[self.STATUS_REGISTRATION_PENDING],
                'organization_id': organization_id,
                'invitation_code': invitation_code,
                'expires_at': int((datetime.utcnow() + timedelta(hours=expires_hours)).timestamp())
            }
            
        except Exception as e:
            _logger.error(f"生成注册令牌失败: {str(e)}")
            raise ValidationError(_('生成注册令牌失败: %s') % str(e))

    @api.model
    def verify_token(self, token):
        """验证JWT令牌
        
        Args:
            token (str): JWT令牌
            
        Returns:
            dict: 验证结果，包含valid字段和解码后的数据
        """
        try:
            secret = self.get_jwt_secret()
            payload = jwt.decode(token, secret, algorithms=['HS256'])
            
            # 添加状态描述
            status = payload.get('status')
            payload['status_description'] = self.STATUS_DESCRIPTIONS.get(status, '未知状态')
            payload['valid'] = True
            payload['expires_at'] = int(payload.get('exp', 0))
            
            _logger.info(f"JWT令牌验证成功: {payload.get('phone')}, 状态: {status}")
            
            return payload
            
        except jwt.ExpiredSignatureError:
            _logger.warning("JWT令牌已过期")
            return {'valid': False, 'error': 'token_expired', 'message': '令牌已过期'}
        except jwt.InvalidTokenError as e:
            _logger.warning(f"JWT令牌无效: {str(e)}")
            return {'valid': False, 'error': 'invalid_token', 'message': '令牌无效'}
        except Exception as e:
            _logger.error(f"JWT令牌验证失败: {str(e)}")
            return {'valid': False, 'error': 'verification_failed', 'message': '令牌验证失败'}

    @api.model
    def _generate_token(self, phone, status, user_data=None, organization_id=None, invitation_code=None, expires_hours=24):
        """内部方法：生成JWT令牌
        
        Args:
            phone (str): 手机号
            status (str): 状态
            user_data (dict): 用户数据
            organization_id (int): 组织ID
            invitation_code (str): 邀请码
            expires_hours (int): 过期小时数
            
        Returns:
            str: JWT令牌
        """
        try:
            # 准备载荷
            payload = {
                'phone': phone,
                'status': status,
                'iat': datetime.utcnow(),
                'exp': datetime.utcnow() + timedelta(hours=expires_hours)
            }
            
            # 添加组织信息
            if organization_id:
                payload['organization_id'] = organization_id
            
            # 添加邀请码信息
            if invitation_code:
                payload['invitation_code'] = invitation_code
            
            # 如果是登录成功，添加用户信息
            if status == self.STATUS_LOGIN_SUCCESS and user_data:
                payload.update({
                    'user_id': user_data.get('id'),
                    'user_name': user_data.get('name'),
                    'user_email': user_data.get('email'),
                })
            
            # 生成JWT令牌
            secret = self.get_jwt_secret()
            token = jwt.encode(payload, secret, algorithm='HS256')
            
            return token
            
        except Exception as e:
            _logger.error(f"生成JWT令牌失败: {str(e)}")
            raise ValidationError(_('生成令牌失败: %s') % str(e))

    @api.model
    def refresh_token(self, token, extends_hours=24):
        """刷新JWT令牌
        
        Args:
            token (str): 原始令牌
            extends_hours (int): 延长小时数
            
        Returns:
            dict: 新令牌信息或错误信息
        """
        try:
            # 验证原始令牌
            verify_result = self.verify_token(token)
            if not verify_result.get('valid'):
                return verify_result
            
            phone = verify_result['phone']
            status = verify_result['status']
            
            if status == self.STATUS_LOGIN_SUCCESS:
                user_id = verify_result.get('user_id')
                if user_id:
                    return self.generate_login_token(user_id, phone, extends_hours)
            elif status == self.STATUS_REGISTRATION_PENDING:
                organization_id = verify_result.get('organization_id')
                invitation_code = verify_result.get('invitation_code')
                return self.generate_registration_token(phone, organization_id, invitation_code, extends_hours)
            
            return {'valid': False, 'error': 'unsupported_status', 'message': '不支持的令牌状态'}
            
        except Exception as e:
            _logger.error(f"刷新令牌失败: {str(e)}")
            return {'valid': False, 'error': 'refresh_failed', 'message': '刷新令牌失败'}
