# -*- coding: utf-8 -*-

import logging
import random
from datetime import datetime, timedelta
from odoo import api, models, fields, _
from odoo.exceptions import ValidationError

_logger = logging.getLogger(__name__)


class SmsService(models.AbstractModel):
    """短信验证码服务"""
    
    _name = 'apollo.sms.service'
    _description = 'SMS Verification Service'

    # 验证码长度
    CODE_LENGTH = 6
    
    # 验证码有效期（分钟）
    CODE_EXPIRY_MINUTES = 5
    
    # 发送冷却期（秒）
    SEND_COOLDOWN_SECONDS = 120

    @api.model
    def create_verification_code(self, phone, invitation_code, organization_id, ip_address=None, user_agent=None):
        """创建短信验证码记录
        
        Args:
            phone (str): 手机号
            invitation_code (str): 邀请码
            organization_id (int): 组织ID
            ip_address (str): IP地址
            user_agent (str): 用户代理
            
        Returns:
            recordset: 验证码记录
        """
        try:
            # 生成6位数字验证码
            code = self._generate_verification_code()
            
            # 计算过期时间
            expires_at = datetime.now() + timedelta(minutes=self.CODE_EXPIRY_MINUTES)
            
            # 创建验证码记录
            verification_record = self.env['apollo.sms.verification.code'].sudo().create({
                'phone': phone,
                'code': code,
                'invitation_code': invitation_code,
                'organization_id': organization_id,
                'expires_at': expires_at,
                'ip_address': ip_address,
                'user_agent': user_agent,
                'created_by': self.env.user.id if not self.env.user._is_public() else False
            })
            
            _logger.info(f"创建验证码记录成功: {phone}, 邀请码: {invitation_code}")
            
            return verification_record
            
        except Exception as e:
            _logger.error(f"创建验证码记录失败: {str(e)}")
            raise ValidationError(_('创建验证码记录失败: %s') % str(e))

    @api.model
    def send_verification_sms(self, verification_record):
        """发送短信验证码
        
        Args:
            verification_record (recordset): 验证码记录
            
        Returns:
            dict: 发送结果 {'success': bool, 'message': str}
        """
        try:
            phone = verification_record.phone
            code = verification_record.code
            
            # 调用短信发送接口
            success, error_message = self._send_sms(phone, code)
            
            # 更新发送状态
            update_vals = {
                'is_sent': success,
                'send_time': datetime.now()
            }
            
            if not success:
                update_vals['error_message'] = error_message
            
            verification_record.write(update_vals)
            
            if success:
                _logger.info(f"短信验证码发送成功: {phone}")
                return {'success': True, 'message': '验证码发送成功'}
            else:
                _logger.error(f"短信验证码发送失败: {phone}, 错误: {error_message}")
                return {'success': False, 'message': error_message or '发送失败'}
                
        except Exception as e:
            error_msg = f"发送短信验证码异常: {str(e)}"
            _logger.error(error_msg)
            
            # 更新错误信息
            verification_record.write({
                'is_sent': False,
                'error_message': error_msg,
                'send_time': datetime.now()
            })
            
            return {'success': False, 'message': '发送异常，请稍后重试'}

    @api.model
    def verify_sms_code(self, phone, code, invitation_code):
        """验证短信验证码
        
        Args:
            phone (str): 手机号
            code (str): 验证码
            invitation_code (str): 邀请码
            
        Returns:
            dict: 验证结果 {'success': bool, 'message': str, 'record': recordset}
        """
        try:
            # 查找匹配的验证码记录
            verification_record = self.env['apollo.sms.verification.code'].sudo().search([
                ('phone', '=', phone),
                ('code', '=', code),
                ('invitation_code', '=', invitation_code),
                ('is_sent', '=', True),
                ('is_used', '=', False)
            ], order='create_date desc', limit=1)
            
            if not verification_record:
                return {'success': False, 'message': '验证码不正确或已使用'}
            
            # 检查是否过期
            if verification_record.is_expired:
                return {'success': False, 'message': '验证码已过期'}
            
            # 标记为已使用
            verification_record.write({
                'is_used': True,
                'used_at': datetime.now()
            })
            
            _logger.info(f"短信验证码验证成功: {phone}, 邀请码: {invitation_code}")
            
            return {
                'success': True, 
                'message': '验证成功',
                'record': verification_record
            }
            
        except Exception as e:
            error_msg = f"验证短信验证码异常: {str(e)}"
            _logger.error(error_msg)
            return {'success': False, 'message': '验证异常，请稍后重试'}

    @api.model
    def check_send_cooldown(self, phone):
        """检查发送冷却期
        
        Args:
            phone (str): 手机号
            
        Returns:
            dict: 检查结果 {'can_send': bool, 'remaining_seconds': int, 'message': str}
        """
        try:
            # 查找最近的发送记录
            recent_record = self.env['apollo.sms.verification.code'].sudo().search([
                ('phone', '=', phone),
                ('is_sent', '=', True)
            ], order='send_time desc', limit=1)
            
            if not recent_record or not recent_record.send_time:
                return {'can_send': True, 'remaining_seconds': 0, 'message': '可以发送'}
            
            # 计算时间差
            time_diff = datetime.now() - recent_record.send_time
            elapsed_seconds = int(time_diff.total_seconds())
            
            if elapsed_seconds < self.SEND_COOLDOWN_SECONDS:
                remaining_seconds = self.SEND_COOLDOWN_SECONDS - elapsed_seconds
                return {
                    'can_send': False,
                    'remaining_seconds': remaining_seconds,
                    'message': f'请等待{remaining_seconds}秒后再次发送'
                }
            
            return {'can_send': True, 'remaining_seconds': 0, 'message': '可以发送'}
            
        except Exception as e:
            _logger.error(f"检查发送冷却期异常: {str(e)}")
            return {'can_send': True, 'remaining_seconds': 0, 'message': '可以发送'}

    @api.model
    def get_sms_statistics(self, date_from=None, date_to=None, organization_id=None):
        """获取短信发送统计
        
        Args:
            date_from (datetime): 开始时间
            date_to (datetime): 结束时间
            organization_id (int): 组织ID
            
        Returns:
            dict: 统计信息
        """
        try:
            domain = []
            
            if date_from:
                domain.append(('create_date', '>=', date_from))
            if date_to:
                domain.append(('create_date', '<=', date_to))
            if organization_id:
                domain.append(('organization_id', '=', organization_id))
            
            records = self.env['apollo.sms.verification.code'].sudo().search(domain)
            
            total_count = len(records)
            sent_count = len(records.filtered('is_sent'))
            failed_count = total_count - sent_count
            used_count = len(records.filtered('is_used'))
            expired_count = len(records.filtered('is_expired'))
            
            return {
                'total_count': total_count,
                'sent_count': sent_count,
                'failed_count': failed_count,
                'used_count': used_count,
                'expired_count': expired_count,
                'success_rate': round((sent_count / total_count * 100), 2) if total_count > 0 else 0,
                'usage_rate': round((used_count / sent_count * 100), 2) if sent_count > 0 else 0
            }
            
        except Exception as e:
            _logger.error(f"获取短信统计异常: {str(e)}")
            return {
                'total_count': 0,
                'sent_count': 0,
                'failed_count': 0,
                'used_count': 0,
                'expired_count': 0,
                'success_rate': 0,
                'usage_rate': 0
            }

    @api.model
    def _generate_verification_code(self):
        """生成验证码
        
        Returns:
            str: 6位数字验证码
        """
        return ''.join([str(random.randint(0, 9)) for _ in range(self.CODE_LENGTH)])

    @api.model
    def _send_sms(self, phone, code):
        """发送短信的内部方法
        
        Args:
            phone (str): 手机号
            code (str): 验证码
            
        Returns:
            tuple: (success: bool, error_message: str or None)
        """
        try:
            # 获取短信服务提供商配置
            provider = self.env['ir.config_parameter'].sudo().get_param(
                'apollo.sms.provider', 'mock'
            )
            
            if provider == 'aliyun':
                return self._send_via_aliyun(phone, code)
            elif provider == 'tencent':
                return self._send_via_tencent(phone, code)
            elif provider == 'custom':
                return self._send_via_custom_provider(phone, code)
            else:
                # 默认使用模拟发送
                return self._send_mock(phone, code)
                
        except Exception as e:
            error_msg = f"发送短信异常: {str(e)}"
            _logger.error(error_msg)
            return False, error_msg

    @api.model
    def _send_mock(self, phone, code):
        """模拟发送短信
        
        Args:
            phone (str): 手机号
            code (str): 验证码
            
        Returns:
            tuple: (success: bool, error_message: str or None)
        """
        _logger.info(f"模拟发送验证码短信到 {phone}: {code}")
        return True, None

    @api.model
    def _send_via_aliyun(self, phone, code):
        """通过阿里云发送短信
        
        Args:
            phone (str): 手机号
            code (str): 验证码
            
        Returns:
            tuple: (success: bool, error_message: str or None)
        """
        try:
            # TODO: 集成阿里云短信服务
            # 需要安装: pip install alibabacloud_dysmsapi20170525
            _logger.info(f"阿里云短信发送模拟: {phone} - {code}")
            return True, None
            
        except Exception as e:
            return False, f"阿里云短信发送失败: {str(e)}"

    @api.model
    def _send_via_tencent(self, phone, code):
        """通过腾讯云发送短信
        
        Args:
            phone (str): 手机号
            code (str): 验证码
            
        Returns:
            tuple: (success: bool, error_message: str or None)
        """
        try:
            # TODO: 集成腾讯云短信服务
            # 需要安装: pip install tencentcloud-sdk-python
            _logger.info(f"腾讯云短信发送模拟: {phone} - {code}")
            return True, None
            
        except Exception as e:
            return False, f"腾讯云短信发送失败: {str(e)}"

    @api.model
    def _send_via_custom_provider(self, phone, code):
        """通过自定义服务商发送短信
        
        Args:
            phone (str): 手机号
            code (str): 验证码
            
        Returns:
            tuple: (success: bool, error_message: str or None)
        """
        try:
            # TODO: 集成自定义短信服务商
            _logger.info(f"自定义短信服务发送模拟: {phone} - {code}")
            return True, None
            
        except Exception as e:
            return False, f"自定义短信发送失败: {str(e)}"
