# -*- coding: utf-8 -*-

import uuid
import logging
from datetime import datetime, timedelta

_logger = logging.getLogger(__name__)


class CodeGenerator:
    """代码生成器"""
    
    @staticmethod
    def generate_invitation_code(length=12):
        """生成邀请码
        
        Args:
            length (int): 代码长度
            
        Returns:
            str: 邀请码
        """
        return str(uuid.uuid4()).replace('-', '').upper()[:length]
    
    @staticmethod
    def generate_verification_code(length=6):
        """生成数字验证码
        
        Args:
            length (int): 代码长度
            
        Returns:
            str: 验证码
        """
        import random
        return ''.join([str(random.randint(0, 9)) for _ in range(length)])


class DateTimeHelper:
    """日期时间辅助类"""
    
    @staticmethod
    def now():
        """获取当前时间
        
        Returns:
            datetime: 当前时间
        """
        return datetime.now()
    
    @staticmethod
    def add_minutes(dt, minutes):
        """增加分钟数
        
        Args:
            dt (datetime): 基础时间
            minutes (int): 要增加的分钟数
            
        Returns:
            datetime: 新时间
        """
        return dt + timedelta(minutes=minutes)
    
    @staticmethod
    def add_hours(dt, hours):
        """增加小时数
        
        Args:
            dt (datetime): 基础时间
            hours (int): 要增加的小时数
            
        Returns:
            datetime: 新时间
        """
        return dt + timedelta(hours=hours)
    
    @staticmethod
    def add_days(dt, days):
        """增加天数
        
        Args:
            dt (datetime): 基础时间
            days (int): 要增加的天数
            
        Returns:
            datetime: 新时间
        """
        return dt + timedelta(days=days)
    
    @staticmethod
    def is_expired(dt, expiry_minutes=5):
        """检查时间是否已过期
        
        Args:
            dt (datetime): 要检查的时间
            expiry_minutes (int): 过期分钟数
            
        Returns:
            bool: 是否已过期
        """
        if not dt:
            return True
        
        expiry_time = dt + timedelta(minutes=expiry_minutes)
        return datetime.now() > expiry_time
    
    @staticmethod
    def format_datetime(dt, format_str='%Y-%m-%d %H:%M:%S'):
        """格式化日期时间
        
        Args:
            dt (datetime): 日期时间对象
            format_str (str): 格式字符串
            
        Returns:
            str: 格式化后的字符串
        """
        if not dt:
            return ''
        
        return dt.strftime(format_str)
    
    @staticmethod
    def parse_datetime(dt_str, format_str='%Y-%m-%d %H:%M:%S'):
        """解析日期时间字符串
        
        Args:
            dt_str (str): 日期时间字符串
            format_str (str): 格式字符串
            
        Returns:
            datetime or None: 解析后的日期时间对象
        """
        try:
            return datetime.strptime(dt_str, format_str)
        except (ValueError, TypeError):
            return None


class ResponseHelper:
    """响应辅助类"""
    
    @staticmethod
    def success_response(message, data=None, code=200):
        """创建成功响应
        
        Args:
            message (str): 成功消息
            data (dict): 响应数据
            code (int): HTTP状态码
            
        Returns:
            dict: 标准化的成功响应
        """
        response = {
            'success': True,
            'message': message,
            'code': code,
            'timestamp': datetime.now().isoformat()
        }
        if data is not None:
            response['data'] = data
        return response
    
    @staticmethod
    def error_response(message, code=400, details=None, error_type=None):
        """创建错误响应
        
        Args:
            message (str): 错误消息
            code (int): HTTP状态码
            details (dict): 错误详情
            error_type (str): 错误类型
            
        Returns:
            dict: 标准化的错误响应
        """
        response = {
            'success': False,
            'message': message,
            'code': code,
            'timestamp': datetime.now().isoformat()
        }
        if details:
            response['details'] = details
        if error_type:
            response['error_type'] = error_type
        return response
    
    @staticmethod
    def validation_error_response(errors):
        """创建验证错误响应
        
        Args:
            errors (list): 错误列表
            
        Returns:
            dict: 验证错误响应
        """
        return ResponseHelper.error_response(
            message='数据验证失败',
            code=400,
            details={'validation_errors': errors},
            error_type='validation_error'
        )


class SecurityHelper:
    """安全辅助类"""
    
    @staticmethod
    def mask_sensitive_data(data, sensitive_fields=None):
        """掩码敏感数据
        
        Args:
            data (dict): 原始数据
            sensitive_fields (list): 敏感字段列表
            
        Returns:
            dict: 掩码后的数据
        """
        if not sensitive_fields:
            sensitive_fields = ['password', 'token', 'secret', 'key']
        
        if not isinstance(data, dict):
            return data
        
        masked_data = data.copy()
        
        for field in sensitive_fields:
            if field in masked_data:
                value = masked_data[field]
                if isinstance(value, str) and len(value) > 6:
                    masked_data[field] = f"{value[:3]}***{value[-3:]}"
                else:
                    masked_data[field] = "***"
        
        return masked_data
    
    @staticmethod
    def get_client_ip(request):
        """获取客户端IP地址
        
        Args:
            request: HTTP请求对象
            
        Returns:
            str: IP地址
        """
        # 尝试从不同的头部获取真实IP
        headers_to_check = [
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'HTTP_X_FORWARDED',
            'HTTP_X_CLUSTER_CLIENT_IP',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        ]
        
        for header in headers_to_check:
            ip = request.httprequest.environ.get(header)
            if ip:
                # 如果有多个IP，取第一个
                return ip.split(',')[0].strip()
        
        return 'unknown'
    
    @staticmethod
    def get_user_agent(request):
        """获取用户代理字符串
        
        Args:
            request: HTTP请求对象
            
        Returns:
            str: User-Agent字符串
        """
        return request.httprequest.environ.get('HTTP_USER_AGENT', '')


class LogHelper:
    """日志辅助类"""
    
    @staticmethod
    def log_api_request(endpoint, method, data=None, user_id=None, ip_address=None):
        """记录API请求日志
        
        Args:
            endpoint (str): API端点
            method (str): HTTP方法
            data (dict): 请求数据
            user_id (int): 用户ID
            ip_address (str): IP地址
        """
        log_data = {
            'endpoint': endpoint,
            'method': method,
            'user_id': user_id,
            'ip_address': ip_address,
            'timestamp': datetime.now().isoformat()
        }
        
        if data:
            # 掩码敏感数据
            log_data['data'] = SecurityHelper.mask_sensitive_data(data)
        
        _logger.info(f"API请求: {log_data}")
    
    @staticmethod
    def log_api_response(endpoint, status_code, response_time=None, error=None):
        """记录API响应日志
        
        Args:
            endpoint (str): API端点
            status_code (int): HTTP状态码
            response_time (float): 响应时间（毫秒）
            error (str): 错误信息
        """
        log_data = {
            'endpoint': endpoint,
            'status_code': status_code,
            'timestamp': datetime.now().isoformat()
        }
        
        if response_time:
            log_data['response_time_ms'] = response_time
        
        if error:
            log_data['error'] = error
            _logger.error(f"API响应错误: {log_data}")
        else:
            _logger.info(f"API响应: {log_data}")


class DataHelper:
    """数据处理辅助类"""
    
    @staticmethod
    def safe_get(data, key, default=None):
        """安全获取字典值
        
        Args:
            data (dict): 数据字典
            key (str): 键名
            default: 默认值
            
        Returns:
            任意类型: 值或默认值
        """
        if not isinstance(data, dict):
            return default
        
        return data.get(key, default)
    
    @staticmethod
    def clean_dict(data, remove_none=True, remove_empty_string=True):
        """清理字典数据
        
        Args:
            data (dict): 原始数据
            remove_none (bool): 是否移除None值
            remove_empty_string (bool): 是否移除空字符串
            
        Returns:
            dict: 清理后的数据
        """
        if not isinstance(data, dict):
            return data
        
        cleaned = {}
        
        for key, value in data.items():
            if remove_none and value is None:
                continue
            if remove_empty_string and value == '':
                continue
            
            cleaned[key] = value
        
        return cleaned
    
    @staticmethod
    def merge_dicts(*dicts):
        """合并多个字典
        
        Args:
            *dicts: 要合并的字典
            
        Returns:
            dict: 合并后的字典
        """
        result = {}
        for d in dicts:
            if isinstance(d, dict):
                result.update(d)
        return result
