from odoo import api, fields, models, _
from odoo.exceptions import UserError


class ApolloOrganizationMemberWizard(models.TransientModel):
    _name = 'apollo.organization.member.wizard'
    _description = 'Apollo Organization Member Management Wizard'

    organization_id = fields.Many2one(
        'apollo.organization',
        string='组织',
        required=True,
        readonly=True
    )
    action = fields.Selection([
        ('add', '添加成员'),
        ('remove', '移除成员')
    ], string='操作类型', required=True, default='add')
    
    user_id = fields.Many2one(
        'res.users',
        string='用户',
        required=True,
        domain=[('active', '=', True)]
    )
    member_id = fields.Many2one(
        'apollo.organization.member',
        string='成员记录',
        domain="[('organization_id', '=', organization_id), ('leave_date', '=', False)]"
    )
    note = fields.Text(
        string='备注',
        help='添加或移除成员的原因'
    )

    @api.onchange('action')
    def _onchange_action(self):
        """根据操作类型调整字段显示"""
        if self.action == 'remove':
            self.user_id = False
        else:
            self.member_id = False

    @api.onchange('member_id')
    def _onchange_member_id(self):
        """当选择成员记录时，自动填充用户"""
        if self.member_id:
            self.user_id = self.member_id.user_id

    def confirm_action(self):
        """确认执行操作"""
        self.ensure_one()
        
        if self.action == 'add':
            return self._add_member()
        elif self.action == 'remove':
            return self._remove_member()

    def _add_member(self):
        """添加成员"""
        if not self.user_id:
            raise UserError(_('请选择要添加的用户'))
        
        # 检查用户是否已经在组织中
        existing_active = self.env['apollo.organization.member'].search([
            ('organization_id', '=', self.organization_id.id),
            ('user_id', '=', self.user_id.id),
            ('leave_date', '=', False)
        ])
        
        if existing_active:
            raise UserError(_('用户 %s 已经在组织中') % self.user_id.name)
        
        # 添加成员
        member = self.env['apollo.organization.member'].add_member(
            self.organization_id.id,
            self.user_id.id,
            self.note
        )
        
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': _('成功'),
                'message': _('用户 %s 已成功添加到组织') % self.user_id.name,
                'type': 'success',
                'sticky': False,
            }
        }

    def _remove_member(self):
        """移除成员"""
        if not self.member_id:
            raise UserError(_('请选择要移除的成员'))
        
        if self.member_id.leave_date:
            raise UserError(_('该成员已经离开组织'))
        
        # 检查是否为组织所有者
        if (self.member_id.organization_id.owner_id and 
            self.member_id.user_id.id == self.member_id.organization_id.owner_id.id):
            raise UserError(_('不能移除组织所有者，请先转移所有权'))
        
        # 移除成员
        success = self.member_id.remove_member(self.note)
        
        if success:
            return {
                'type': 'ir.actions.client',
                'tag': 'display_notification',
                'params': {
                    'title': _('成功'),
                    'message': _('用户 %s 已从组织中移除') % self.member_id.user_id.name,
                    'type': 'success',
                    'sticky': False,
                }
            }
        else:
            raise UserError(_('移除成员失败'))
