/**
 * Apollo Frontend JavaScript
 * 
 * 提供与 Apollo Account API 的前端交互功能
 */

(function () {
    'use strict';

    // API 基础配置
    const API_BASE = '/api/apollo';
    const STORAGE_KEY = 'apollo_token';
    const STORAGE_USER_KEY = 'apollo_user_info';

    // 工具函数
    const Utils = {
        // 显示消息
        showMessage: function (message, type = 'info') {
            const messageEl = document.getElementById('apollo-message');
            if (messageEl) {
                messageEl.className = `apollo-message ${type}`;
                messageEl.textContent = message;
                messageEl.style.display = 'block';

                // 3秒后自动隐藏成功/信息消息
                if (type === 'success' || type === 'info') {
                    setTimeout(() => {
                        messageEl.style.display = 'none';
                    }, 3000);
                }
            }
        },

        // 隐藏消息
        hideMessage: function () {
            const messageEl = document.getElementById('apollo-message');
            if (messageEl) {
                messageEl.style.display = 'none';
            }
        },

        // API 请求
        apiRequest: async function (endpoint, data = {}, showLoading = true) {
            try {
                if (showLoading) {
                    this.showMessage('请求中...', 'info');
                }

                const response = await fetch(`${API_BASE}${endpoint}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(data)
                });

                const result = await response.json();

                if (showLoading) {
                    this.hideMessage();
                }

                if (!result.success) {
                    throw new Error(result.message || '请求失败');
                }

                return result;
            } catch (error) {
                if (showLoading) {
                    this.hideMessage();
                }
                this.showMessage(error.message || '网络错误，请稍后重试', 'error');
                throw error;
            }
        },

        // 存储 Token
        saveToken: function (token) {
            localStorage.setItem(STORAGE_KEY, token);
        },

        // 获取 Token
        getToken: function () {
            return localStorage.getItem(STORAGE_KEY);
        },

        // 清除 Token
        clearToken: function () {
            localStorage.removeItem(STORAGE_KEY);
            localStorage.removeItem(STORAGE_USER_KEY);
        },

        // 存储用户信息
        saveUserInfo: function (userInfo) {
            localStorage.setItem(STORAGE_USER_KEY, JSON.stringify(userInfo));
        },

        // 获取用户信息
        getUserInfo: function () {
            const userInfo = localStorage.getItem(STORAGE_USER_KEY);
            return userInfo ? JSON.parse(userInfo) : null;
        },

        // 格式化手机号
        formatPhone: function (phone) {
            if (!phone) return '';
            const cleaned = phone.replace(/\D/g, '');
            if (cleaned.length === 11 && cleaned.startsWith('1')) {
                return cleaned.replace(/(\d{3})(\d{4})(\d{4})/, '$1****$3');
            }
            return phone;
        },

        // 格式化日期
        formatDate: function (dateString) {
            if (!dateString) return '';
            const date = new Date(dateString);
            return date.toLocaleDateString('zh-CN');
        },

        // 获取姓名首字母
        getInitials: function (name) {
            if (!name) return '?';
            const words = name.trim().split(' ');
            if (words.length >= 2) {
                return (words[0][0] + words[1][0]).toUpperCase();
            }
            return name[0].toUpperCase();
        }
    };

    // 登录页面逻辑
    const LoginPage = {
        init: function () {
            // 检查是否已登录
            if (Utils.getToken()) {
                window.location.href = '/apollo/profile';
                return;
            }

            this.bindEvents();
        },

        bindEvents: function () {
            // 标签页切换
            const tabBtns = document.querySelectorAll('.apollo-tab-btn');
            const tabContents = document.querySelectorAll('.apollo-tab-content');

            tabBtns.forEach(btn => {
                btn.addEventListener('click', (e) => {
                    const tabName = e.target.getAttribute('data-tab');

                    // 更新标签按钮状态
                    tabBtns.forEach(b => b.classList.remove('active'));
                    e.target.classList.add('active');

                    // 更新内容显示
                    tabContents.forEach(content => {
                        content.classList.remove('active');
                    });
                    document.getElementById(`${tabName}-login`).classList.add('active');
                });
            });

            // 密码登录表单
            const passwordForm = document.getElementById('passwordLoginForm');
            if (passwordForm) {
                passwordForm.addEventListener('submit', this.handlePasswordLogin.bind(this));
            }

            // 短信登录表单
            const smsForm = document.getElementById('smsLoginForm');
            if (smsForm) {
                smsForm.addEventListener('submit', this.handleSmsLogin.bind(this));
            }

            // 发送短信验证码
            const sendSmsBtn = document.getElementById('sendSmsBtn');
            if (sendSmsBtn) {
                sendSmsBtn.addEventListener('click', this.handleSendLoginSms.bind(this));
            }
        },

        handlePasswordLogin: async function (e) {
            e.preventDefault();

            const formData = new FormData(e.target);
            const phone = formData.get('phone');
            const password = formData.get('password');

            if (!phone || !password) {
                Utils.showMessage('请填写完整信息', 'error');
                return;
            }

            try {
                const data = { password };

                // 判断是手机号还是邮箱
                if (phone.includes('@')) {
                    data.email = phone;
                } else {
                    data.phone = phone;
                }

                const result = await Utils.apiRequest('/user/login', data);

                if (result.data && result.data.token) {
                    Utils.saveToken(result.data.token);
                    Utils.showMessage('登录成功！', 'success');
                    setTimeout(() => {
                        window.location.href = '/apollo/profile';
                    }, 1000);
                } else {
                    Utils.showMessage('登录失败，请检查用户名和密码', 'error');
                }
            } catch (error) {
                // Utils.apiRequest 已经显示了错误消息
            }
        },

        handleSmsLogin: async function (e) {
            e.preventDefault();

            const formData = new FormData(e.target);
            const phone = formData.get('phone');
            const smsCode = formData.get('sms_code');

            if (!phone || !smsCode) {
                Utils.showMessage('请填写完整信息', 'error');
                return;
            }

            try {
                const result = await Utils.apiRequest('/user/login/sms', {
                    phone: phone,
                    sms_code: smsCode
                });

                if (result.data && result.data.token) {
                    Utils.saveToken(result.data.token);
                    Utils.showMessage('登录成功！', 'success');
                    setTimeout(() => {
                        window.location.href = '/apollo/profile';
                    }, 1000);
                } else {
                    Utils.showMessage('登录失败，请检查验证码', 'error');
                }
            } catch (error) {
                // Utils.apiRequest 已经显示了错误消息
            }
        },

        handleSendLoginSms: async function (e) {
            e.preventDefault();

            const phoneInput = document.getElementById('sms-phone');
            const phone = phoneInput.value.trim();

            if (!phone) {
                Utils.showMessage('请输入手机号', 'error');
                return;
            }

            const btn = e.target;
            const originalText = btn.textContent;

            try {
                btn.disabled = true;
                btn.textContent = '发送中...';

                await Utils.apiRequest('/sms/send_login_code', { phone: phone });

                Utils.showMessage('验证码发送成功', 'success');

                // 倒计时
                this.startCountdown(btn, 60);
            } catch (error) {
                btn.disabled = false;
                btn.textContent = originalText;
            }
        },

        startCountdown: function (btn, seconds) {
            let remaining = seconds;
            const originalText = '发送验证码';

            const timer = setInterval(() => {
                btn.textContent = `${remaining}秒后重发`;
                remaining--;

                if (remaining < 0) {
                    clearInterval(timer);
                    btn.disabled = false;
                    btn.textContent = originalText;
                }
            }, 1000);
        }
    };

    // 注册页面逻辑
    const RegisterPage = {
        currentStep: 1,
        invitationData: null,
        registrationToken: null,

        init: function () {
            // 检查是否已登录
            if (Utils.getToken()) {
                window.location.href = '/apollo/profile';
                return;
            }

            // 检查URL参数中的邀请码
            const urlParams = new URLSearchParams(window.location.search);
            const invitationCode = urlParams.get('invitation_code');
            if (invitationCode) {
                const input = document.getElementById('invitation-code');
                if (input) {
                    input.value = invitationCode;
                }
            }

            this.bindEvents();
        },

        bindEvents: function () {
            // 步骤1：验证邀请码
            const invitationForm = document.getElementById('invitationForm');
            if (invitationForm) {
                invitationForm.addEventListener('submit', this.handleInvitationVerify.bind(this));
            }

            // 步骤2：短信验证
            const smsVerifyForm = document.getElementById('smsVerifyForm');
            if (smsVerifyForm) {
                smsVerifyForm.addEventListener('submit', this.handleSmsVerify.bind(this));
            }

            // 发送邀请注册验证码
            const sendInvitationSmsBtn = document.getElementById('sendInvitationSmsBtn');
            if (sendInvitationSmsBtn) {
                sendInvitationSmsBtn.addEventListener('click', this.handleSendInvitationSms.bind(this));
            }

            // 步骤3：完成注册
            const registerForm = document.getElementById('registerForm');
            if (registerForm) {
                registerForm.addEventListener('submit', this.handleRegister.bind(this));
            }
        },

        handleInvitationVerify: async function (e) {
            e.preventDefault();

            const formData = new FormData(e.target);
            const invitationCode = formData.get('invitation_code');

            if (!invitationCode) {
                Utils.showMessage('请输入邀请码', 'error');
                return;
            }

            try {
                const result = await Utils.apiRequest('/invitation/info', {
                    invitation_code: invitationCode
                });

                if (result.data) {
                    this.invitationData = result.data;

                    // 显示绑定的手机号
                    const phoneEl = document.getElementById('bound-phone');
                    if (phoneEl) {
                        phoneEl.textContent = Utils.formatPhone(result.data.phone);
                    }

                    Utils.showMessage('邀请码验证成功', 'success');
                    this.goToStep(2);
                } else {
                    Utils.showMessage('邀请码无效', 'error');
                }
            } catch (error) {
                // Utils.apiRequest 已经显示了错误消息
            }
        },

        handleSendInvitationSms: async function (e) {
            e.preventDefault();

            if (!this.invitationData) {
                Utils.showMessage('请先验证邀请码', 'error');
                return;
            }

            const btn = e.target;
            const originalText = btn.textContent;

            try {
                btn.disabled = true;
                btn.textContent = '发送中...';

                await Utils.apiRequest('/sms/send_invitation_code', {
                    invitation_code: this.invitationData.code
                });

                Utils.showMessage('验证码发送成功', 'success');

                // 倒计时
                this.startCountdown(btn, 60);
            } catch (error) {
                btn.disabled = false;
                btn.textContent = originalText;
            }
        },

        handleSmsVerify: async function (e) {
            e.preventDefault();

            const formData = new FormData(e.target);
            const smsCode = formData.get('sms_code');

            if (!smsCode) {
                Utils.showMessage('请输入验证码', 'error');
                return;
            }

            if (!this.invitationData) {
                Utils.showMessage('请先验证邀请码', 'error');
                return;
            }

            try {
                const result = await Utils.apiRequest('/sms/verify_invitation_code', {
                    invitation_code: this.invitationData.code,
                    sms_code: smsCode
                });

                if (result.data && result.data.token) {
                    this.registrationToken = result.data.token;
                    Utils.showMessage('短信验证成功', 'success');
                    this.goToStep(3);
                } else {
                    Utils.showMessage('验证码错误', 'error');
                }
            } catch (error) {
                // Utils.apiRequest 已经显示了错误消息
            }
        },

        handleRegister: async function (e) {
            e.preventDefault();

            const formData = new FormData(e.target);
            const name = formData.get('name');
            const password = formData.get('password');
            const confirmPassword = formData.get('confirm_password');

            if (!name || !password || !confirmPassword) {
                Utils.showMessage('请填写完整信息', 'error');
                return;
            }

            if (password !== confirmPassword) {
                Utils.showMessage('两次输入的密码不一致', 'error');
                return;
            }

            if (password.length < 6) {
                Utils.showMessage('密码长度至少6位', 'error');
                return;
            }

            if (!this.registrationToken) {
                Utils.showMessage('注册令牌无效，请重新验证', 'error');
                return;
            }

            try {
                const result = await Utils.apiRequest('/user/register', {
                    token: this.registrationToken,
                    name: name,
                    password: password
                });

                if (result.data && result.data.token) {
                    Utils.saveToken(result.data.token);
                    Utils.showMessage('注册成功！正在跳转...', 'success');
                    setTimeout(() => {
                        window.location.href = '/apollo/profile';
                    }, 2000);
                } else {
                    Utils.showMessage('注册失败，请稍后重试', 'error');
                }
            } catch (error) {
                // Utils.apiRequest 已经显示了错误消息
            }
        },

        goToStep: function (step) {
            // 更新步骤指示器
            const steps = document.querySelectorAll('.apollo-step');
            const contents = document.querySelectorAll('.apollo-step-content');

            steps.forEach((stepEl, index) => {
                const stepNum = index + 1;
                stepEl.classList.remove('active', 'completed');

                if (stepNum < step) {
                    stepEl.classList.add('completed');
                } else if (stepNum === step) {
                    stepEl.classList.add('active');
                }
            });

            // 更新内容显示
            contents.forEach((content, index) => {
                const stepNum = index + 1;
                content.classList.remove('active');

                if (stepNum === step) {
                    content.classList.add('active');
                }
            });

            this.currentStep = step;
        },

        startCountdown: function (btn, seconds) {
            let remaining = seconds;
            const originalText = '发送验证码';

            const timer = setInterval(() => {
                btn.textContent = `${remaining}秒后重发`;
                remaining--;

                if (remaining < 0) {
                    clearInterval(timer);
                    btn.disabled = false;
                    btn.textContent = originalText;
                }
            }, 1000);
        }
    };

    // 用户信息页面逻辑
    const ProfilePage = {
        init: function () {
            const token = Utils.getToken();
            if (!token) {
                window.location.href = '/apollo/login';
                return;
            }

            this.bindEvents();
            this.loadUserProfile();
        },

        bindEvents: function () {
            // 退出登录
            const logoutBtn = document.getElementById('logoutBtn');
            if (logoutBtn) {
                logoutBtn.addEventListener('click', this.handleLogout.bind(this));
            }
        },

        handleLogout: function () {
            if (confirm('确认退出登录吗？')) {
                Utils.clearToken();
                Utils.showMessage('已退出登录', 'info');
                setTimeout(() => {
                    window.location.href = '/apollo/login';
                }, 1000);
            }
        },

        loadUserProfile: async function () {
            const token = Utils.getToken();
            const loadingEl = document.getElementById('loading-state');
            const contentEl = document.getElementById('profile-content');
            const errorEl = document.getElementById('error-state');

            try {
                // 显示加载状态
                if (loadingEl) loadingEl.style.display = 'block';
                if (contentEl) contentEl.style.display = 'none';
                if (errorEl) errorEl.style.display = 'none';

                const result = await Utils.apiRequest('/user/profile', { token: token }, false);

                if (result.data) {
                    this.displayUserInfo(result.data);
                    Utils.saveUserInfo(result.data);

                    // 显示内容
                    if (loadingEl) loadingEl.style.display = 'none';
                    if (contentEl) contentEl.style.display = 'block';
                } else {
                    throw new Error('无法获取用户信息');
                }
            } catch (error) {
                console.error('Load profile error:', error);

                // 显示错误状态
                if (loadingEl) loadingEl.style.display = 'none';
                if (contentEl) contentEl.style.display = 'none';
                if (errorEl) {
                    errorEl.style.display = 'block';
                    const errorMsgEl = document.getElementById('error-message');
                    if (errorMsgEl) {
                        errorMsgEl.textContent = error.message || '获取用户信息失败';
                    }
                }
            }
        },

        displayUserInfo: function (data) {
            // 显示用户头像
            const avatarEl = document.getElementById('user-avatar');
            if (avatarEl && data.user) {
                avatarEl.textContent = Utils.getInitials(data.user.name);
            }

            // 显示用户信息
            if (data.user) {
                const nameEl = document.getElementById('user-name');
                if (nameEl) nameEl.textContent = data.user.name || '未设置';

                const phoneEl = document.getElementById('user-phone');
                if (phoneEl) phoneEl.textContent = data.user.phone || '未绑定';

                const emailEl = document.getElementById('user-email');
                if (emailEl) emailEl.textContent = data.user.email || '未绑定';
            }

            // 显示组织信息
            if (data.organization) {
                const orgLogoEl = document.getElementById('org-logo');
                if (orgLogoEl) {
                    orgLogoEl.textContent = data.organization.name ? data.organization.name[0].toUpperCase() : 'O';
                }

                const orgNameEl = document.getElementById('org-name');
                if (orgNameEl) orgNameEl.textContent = data.organization.name || '未知组织';

                const orgIdEl = document.getElementById('org-id');
                if (orgIdEl) orgIdEl.textContent = data.organization.id || '-';
            }

            // 显示成员信息
            if (data.member) {
                const memberRoleEl = document.getElementById('member-role');
                if (memberRoleEl) memberRoleEl.textContent = data.member.role || '成员';

                const joinDateEl = document.getElementById('join-date');
                if (joinDateEl) joinDateEl.textContent = Utils.formatDate(data.member.create_date) || '-';
            }

            // 显示Token信息
            const token = Utils.getToken();
            if (token) {
                const tokenEl = document.getElementById('jwt-token');
                if (tokenEl) tokenEl.value = token;

                const statusEl = document.getElementById('token-status');
                if (statusEl) {
                    statusEl.textContent = '有效';
                    statusEl.className = 'apollo-status-badge valid';
                }

                // 尝试解析Token过期时间
                try {
                    const payload = JSON.parse(atob(token.split('.')[1]));
                    const expiresEl = document.getElementById('token-expires');
                    if (expiresEl && payload.exp) {
                        const expireDate = new Date(payload.exp * 1000);
                        expiresEl.textContent = expireDate.toLocaleString('zh-CN');
                    }
                } catch (e) {
                    console.warn('无法解析Token过期时间');
                }
            }
        }
    };

    // 页面初始化
    document.addEventListener('DOMContentLoaded', function () {
        // 应用基础样式类
        document.body.classList.add('apollo-body');

        // 根据页面类型初始化相应逻辑
        const pageType = document.body.getAttribute('data-page-type') ||
            document.querySelector('[data-page-type]')?.getAttribute('data-page-type');

        switch (pageType) {
            case 'login':
                LoginPage.init();
                break;
            case 'register':
                RegisterPage.init();
                break;
            case 'profile':
                ProfilePage.init();
                break;
            default:
                // 默认检查登录状态
                if (Utils.getToken()) {
                    if (window.location.pathname === '/apollo' || window.location.pathname === '/apollo/') {
                        window.location.href = '/apollo/profile';
                    }
                }
                break;
        }
    });

    // 暴露工具函数到全局（可选）
    window.ApolloUtils = Utils;

})();
