# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import ValidationError


class ApolloOrganizationExtension(models.Model):
    """扩展组织模型，添加套餐相关功能"""
    _inherit = 'apollo.organization'

    # 套餐订阅相关字段
    package_subscription_ids = fields.One2many(
        'apollo.organization.package.subscription', 
        'organization_id', 
        string='套餐订阅记录'
    )
    active_package_subscription_ids = fields.One2many(
        'apollo.organization.package.subscription',
        'organization_id',
        string='当前套餐订阅',
        domain=[('status', '=', 'active')]
    )
    package_subscription_count = fields.Integer(
        string='套餐订阅数量',
        compute='_compute_package_subscription_count'
    )
    current_package_id = fields.Many2one(
        'apollo.package',
        string='当前套餐',
        compute='_compute_current_package_id',
        help='当前生效的套餐'
    )
    current_subscription_end_date = fields.Date(
        string='当前订阅结束日期',
        compute='_compute_current_subscription_info',
        help='当前套餐订阅的结束日期'
    )
    current_subscription_status = fields.Selection([
        ('active', '激活'),
        ('expiring_soon', '即将过期'),
        ('expired', '已过期'),
        ('none', '无套餐'),
    ], string='订阅状态', compute='_compute_current_subscription_info')

    @api.depends('package_subscription_ids')
    def _compute_package_subscription_count(self):
        """计算套餐订阅数量"""
        for record in self:
            record.package_subscription_count = len(record.package_subscription_ids)

    @api.depends('active_package_subscription_ids.is_active_subscription')
    def _compute_current_package_id(self):
        """计算当前套餐"""
        for record in self:
            active_subs = record.active_package_subscription_ids.filtered('is_active_subscription')
            if active_subs:
                # 如果有多个活跃订阅，取最新的
                latest_sub = active_subs.sorted('start_date', reverse=True)[0]
                record.current_package_id = latest_sub.package_id
            else:
                record.current_package_id = False

    @api.depends('active_package_subscription_ids.final_end_date')
    def _compute_current_subscription_info(self):
        """计算当前订阅信息"""
        today = fields.Date.today()
        for record in self:
            active_subs = record.active_package_subscription_ids.filtered('is_active_subscription')
            if active_subs:
                latest_sub = active_subs.sorted('start_date', reverse=True)[0]
                record.current_subscription_end_date = latest_sub.final_end_date
                
                # 计算订阅状态
                if latest_sub.final_end_date:
                    days_remaining = (latest_sub.final_end_date - today).days
                    if days_remaining > 30:
                        record.current_subscription_status = 'active'
                    elif days_remaining > 0:
                        record.current_subscription_status = 'expiring_soon'
                    else:
                        record.current_subscription_status = 'expired'
                else:
                    record.current_subscription_status = 'expired'
            else:
                record.current_subscription_end_date = False
                record.current_subscription_status = 'none'

    def action_configure_package_subscription(self):
        """配置套餐订阅"""
        return {
            'name': _('配置套餐订阅'),
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.package.config.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_organization_id': self.id,
            },
        }

    def action_view_package_subscriptions(self):
        """查看套餐订阅"""
        return {
            'name': f'{self.name} - 套餐订阅',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.package.subscription',
            'view_mode': 'list,form',
            'domain': [('organization_id', '=', self.id)],
            'context': {
                'default_organization_id': self.id,
                'search_default_active': 1,
            },
        }

    def action_view_quota_usage(self):
        """查看额度使用情况"""
        return {
            'name': f'{self.name} - 额度使用情况',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.quota.usage',
            'view_mode': 'list,form',
            'domain': [('subscription_id.organization_id', '=', self.id)],
            'context': {'default_organization_id': self.id},
        }

    def action_extend_current_subscription(self):
        """延长当前订阅"""
        active_subs = self.active_package_subscription_ids.filtered('is_active_subscription')
        if not active_subs:
            raise ValidationError(_('当前没有活跃的套餐订阅'))
        
        latest_sub = active_subs.sorted('start_date', reverse=True)[0]
        return latest_sub.action_extend_subscription()

    def action_adjust_quota(self):
        """调整额度"""
        active_subs = self.active_package_subscription_ids.filtered('is_active_subscription')
        if not active_subs:
            raise ValidationError(_('当前没有活跃的套餐订阅'))
        
        latest_sub = active_subs.sorted('start_date', reverse=True)[0]
        return latest_sub.action_adjust_quota()

    def get_current_package_quota(self, quota_code):
        """获取当前套餐的指定额度"""
        if not self.current_package_id:
            return 0
        
        quota = self.current_package_id.quota_ids.filtered(lambda q: q.quota_definition_id.code == quota_code)
        return quota.quota_amount if quota else 0

    def check_quota_permission(self, quota_code, required_amount=1):
        """检查额度权限"""
        if not self.current_package_id:
            return False
        
        # 获取当前套餐的额度定义
        quota = self.current_package_id.quota_ids.filtered(lambda q: q.quota_definition_id.code == quota_code)
        if not quota:
            return False
        
        # 获取当前使用情况
        active_subs = self.active_package_subscription_ids.filtered('is_active_subscription')
        if not active_subs:
            return False
        
        latest_sub = active_subs.sorted('start_date', reverse=True)[0]
        usage = latest_sub.quota_usage_ids.filtered(lambda u: u.quota_definition_id.code == quota_code)
        
        if usage:
            return usage.remaining_amount >= required_amount
        else:
            # 如果没有使用记录，说明还没开始使用，返回套餐额度是否足够
            return quota.quota_amount >= required_amount
