# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError
from datetime import datetime, timedelta


class OrganizationPackageSubscription(models.Model):
    """组织套餐订阅表 - 记录组织的套餐订阅情况"""
    _name = 'apollo.organization.package.subscription'
    _description = '组织套餐订阅'
    _order = 'start_date desc, id'
    _rec_name = 'display_name'

    organization_id = fields.Many2one(
        'apollo.organization',
        string='组织',
        required=True,
        ondelete='cascade',
        help='关联的组织'
    )
    package_id = fields.Many2one(
        'apollo.package',
        string='套餐',
        required=True,
        help='关联的套餐'
    )
    start_date = fields.Date(
        string='开始日期',
        required=True,
        default=fields.Date.today,
        help='订阅开始日期'
    )
    end_date = fields.Date(
        string='结束日期',
        required=True,
        help='订阅结束日期'
    )
    status = fields.Selection([
        ('active', '激活'),
        ('expired', '过期'),
        ('cancelled', '取消'),
    ], string='订阅状态', required=True, default='active', help='订阅状态')
    
    created_by = fields.Many2one(
        'res.users',
        string='创建人',
        required=True,
        default=lambda self: self.env.user,
        help='创建此订阅的用户'
    )
    created_date = fields.Datetime(
        string='创建时间',
        default=fields.Datetime.now,
        readonly=True
    )



    # 计算字段
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    days_remaining = fields.Integer(
        string='剩余天数',
        compute='_compute_days_remaining',
        search='_search_days_remaining'
    )
    final_end_date = fields.Date(
        string='最终结束日期',
        compute='_compute_final_end_date',
        help='考虑延长后的最终结束日期'
    )
    is_active_subscription = fields.Boolean(
        string='是否有效订阅',
        compute='_compute_is_active_subscription',
        search='_search_is_active_subscription'
    )

    # 关联记录
    extension_ids = fields.One2many(
        'apollo.organization.subscription.extension',
        'subscription_id',
        string='延长记录'
    )
    quota_adjustment_ids = fields.One2many(
        'apollo.organization.quota.adjustment',
        'subscription_id',
        string='额度调整记录'
    )
    quota_usage_ids = fields.One2many(
        'apollo.organization.quota.usage',
        'subscription_id',
        string='额度使用记录'
    )

    _sql_constraints = [
        ('start_end_date_check', 'check(start_date <= end_date)', 
         '开始日期不能晚于结束日期！'),
    ]

    @api.depends('organization_id.name', 'package_id.name', 'start_date', 'end_date')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.organization_id and record.package_id:
                record.display_name = f"{record.organization_id.name} - {record.package_id.name} ({record.start_date} ~ {record.end_date})"
            else:
                record.display_name = "新订阅"

    @api.depends('final_end_date')
    def _compute_days_remaining(self):
        """计算剩余天数"""
        today = fields.Date.today()
        for record in self:
            if record.final_end_date:
                delta = record.final_end_date - today
                record.days_remaining = delta.days
            else:
                record.days_remaining = 0

    @api.depends('end_date', 'extension_ids')
    def _compute_final_end_date(self):
        """计算最终结束日期（考虑延长）"""
        for record in self:
            final_date = record.end_date
            active_extensions = record.extension_ids.filtered('is_active')
            if active_extensions:
                # 获取所有有效的延长日期，过滤掉False值
                extension_dates = [date for date in active_extensions.mapped('new_end_date') if date]
                if extension_dates:
                    # 取最新的延长日期
                    final_date = max(extension_dates)
            # 确保final_date是有效的日期，如果为False则使用today作为默认值
            record.final_end_date = final_date if final_date else fields.Date.today()

    @api.depends('status', 'final_end_date')
    def _compute_is_active_subscription(self):
        """计算是否有效订阅"""
        today = fields.Date.today()
        for record in self:
            # 确保final_end_date是有效的日期值再进行比较
            is_date_valid = record.final_end_date and isinstance(record.final_end_date, (type(today), str))
            record.is_active_subscription = (
                record.status == 'active' and 
                is_date_valid and
                record.final_end_date >= today
            )

    def _search_is_active_subscription(self, operator, value):
        """搜索是否有效订阅"""
        today = fields.Date.today()
        if operator == '=' and value:
            return [
                ('status', '=', 'active'),
                ('end_date', '>=', today)
            ]
        elif operator == '=' and not value:
            return [
                '|',
                ('status', '!=', 'active'),
                ('end_date', '<', today)
            ]
        else:
            return []

    def _search_days_remaining(self, operator, value):
        """搜索剩余天数"""
        today = fields.Date.today()
        if operator == '<=' and isinstance(value, int):
            target_date = today + timedelta(days=value)
            return [('end_date', '<=', target_date)]
        elif operator == '<' and isinstance(value, int):
            if value == 0:
                return [('end_date', '<', today)]
            else:
                target_date = today + timedelta(days=value)
                return [('end_date', '<', target_date)]
        elif operator == '>' and isinstance(value, int):
            target_date = today + timedelta(days=value)
            return [('end_date', '>', target_date)]
        else:
            return [('id', '>', 0)]  # 返回所有记录

    @api.onchange('package_id', 'start_date')
    def _onchange_package_duration(self):
        """根据套餐时长自动计算结束日期"""
        if self.package_id and self.start_date and self.package_id.duration_weeks:
            end_date = self.start_date + timedelta(weeks=self.package_id.duration_weeks)
            self.end_date = end_date

    @api.model_create_multi
    def create(self, vals_list):
        """创建时自动计算结束日期"""
        for vals in vals_list:
            if vals.get('package_id') and vals.get('start_date') and not vals.get('end_date'):
                package = self.env['apollo.package'].browse(vals['package_id'])
                if package.duration_weeks:
                    start_date = fields.Date.from_string(vals['start_date'])
                    end_date = start_date + timedelta(weeks=package.duration_weeks)
                    vals['end_date'] = fields.Date.to_string(end_date)
        return super().create(vals_list)

    def action_extend_subscription(self):
        """延长订阅"""
        return {
            'name': '延长订阅',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.subscription.extension',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_subscription_id': self.id,
                'default_original_end_date': self.final_end_date,
            },
        }

    def action_adjust_quota(self):
        """调整额度"""
        return {
            'name': '调整额度',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.quota.adjustment',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_organization_id': self.organization_id.id,
                'default_subscription_id': self.id,
            },
        }

    def action_view_quota_usage(self):
        """查看额度使用情况"""
        return {
            'name': f'{self.display_name} - 额度使用情况',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.quota.usage',
            'view_mode': 'list,form',
            'domain': [('subscription_id', '=', self.id)],
            'context': {'default_subscription_id': self.id},
        }

    def action_cancel_subscription(self):
        """取消订阅"""
        self.status = 'cancelled'

    def action_reactivate_subscription(self):
        """重新激活订阅"""
        today = fields.Date.today()
        if self.final_end_date and self.final_end_date >= today:
            self.status = 'active'

    @api.model
    def _cron_update_expired_subscriptions(self):
        """定时任务：更新过期订阅状态"""
        today = fields.Date.today()
        # 使用end_date字段搜索，避免计算字段在搜索中的问题
        expired_subscriptions = self.search([
            ('status', '=', 'active'),
            ('end_date', '<', today)
        ])
        # 再次过滤，确保真正过期的订阅
        truly_expired = expired_subscriptions.filtered(
            lambda r: r.final_end_date and r.final_end_date < today
        )
        truly_expired.write({'status': 'expired'})
