# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError


class OrganizationQuotaUsage(models.Model):
    """组织额度使用记录表 - 跟踪组织的额度使用情况"""
    _name = 'apollo.organization.quota.usage'
    _description = '组织额度使用记录'
    _order = 'last_updated desc, id'
    _rec_name = 'display_name'

    organization_id = fields.Many2one(
        'apollo.organization',
        string='组织',
        required=True,
        ondelete='cascade',
        help='关联的组织'
    )
    subscription_id = fields.Many2one(
        'apollo.organization.package.subscription',
        string='订阅',
        required=True,
        help='关联的订阅'
    )
    quota_definition_id = fields.Many2one(
        'apollo.quota.definition',
        string='额度定义',
        required=True,
        help='关联的额度定义'
    )
    used_amount = fields.Integer(
        string='已使用数量',
        default=0,
        help='已使用的额度数量'
    )
    last_updated = fields.Datetime(
        string='最后更新时间',
        default=fields.Datetime.now,
        help='最后更新时间'
    )



    # 计算字段
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    quota_limit = fields.Integer(
        string='额度限制',
        compute='_compute_quota_info'
    )
    remaining_amount = fields.Integer(
        string='剩余数量',
        compute='_compute_quota_info'
    )
    usage_percentage = fields.Float(
        string='使用百分比',
        compute='_compute_quota_info',
        search='_search_usage_percentage'
    )
    is_unlimited = fields.Boolean(
        string='是否无限制',
        compute='_compute_quota_info',
        search='_search_is_unlimited'
    )
    is_over_limit = fields.Boolean(
        string='是否超限',
        compute='_compute_quota_info',
        search='_search_is_over_limit'
    )
    usage_status = fields.Selection([
        ('normal', '正常'),
        ('warning', '警告'),
        ('critical', '严重'),
        ('over_limit', '超限'),
    ], string='使用状态', compute='_compute_usage_status', search='_search_usage_status')

    _sql_constraints = [
        ('org_sub_quota_unique', 
         'unique(organization_id, subscription_id, quota_definition_id)', 
         '同一组织订阅中的额度使用记录不能重复！'),
        ('used_amount_positive', 'check(used_amount >= 0)', 
         '已使用数量不能为负数！'),
    ]

    @api.depends('organization_id.name', 'quota_definition_id.quota_name', 'used_amount')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.organization_id and record.quota_definition_id:
                record.display_name = f"{record.organization_id.name} - {record.quota_definition_id.quota_name} (已用 {record.used_amount})"
            else:
                record.display_name = "新使用记录"

    @api.depends('subscription_id', 'quota_definition_id', 'used_amount')
    def _compute_quota_info(self):
        """计算额度信息"""
        for record in self:
            # 获取有效额度限制
            quota_service = self.env['apollo.package.quota.service']
            effective_quotas = quota_service.get_organization_effective_quotas(record.organization_id.id)
            
            quota_code = record.quota_definition_id.quota_code
            if quota_code in effective_quotas:
                quota_info = effective_quotas[quota_code]
                record.quota_limit = quota_info.get('limit', 0)
                record.is_unlimited = quota_info.get('is_unlimited', False)
                
                if record.is_unlimited:
                    record.remaining_amount = 999999  # 表示无限
                    record.usage_percentage = 0.0
                    record.is_over_limit = False
                else:
                    record.remaining_amount = max(0, record.quota_limit - record.used_amount)
                    if record.quota_limit > 0:
                        record.usage_percentage = (record.used_amount / record.quota_limit) * 100
                    else:
                        record.usage_percentage = 100.0 if record.used_amount > 0 else 0.0
                    record.is_over_limit = record.used_amount > record.quota_limit
            else:
                record.quota_limit = 0
                record.is_unlimited = False
                record.remaining_amount = 0
                record.usage_percentage = 100.0 if record.used_amount > 0 else 0.0
                record.is_over_limit = record.used_amount > 0

    @api.depends('usage_percentage', 'is_over_limit', 'is_unlimited')
    def _compute_usage_status(self):
        """计算使用状态"""
        for record in self:
            if record.is_unlimited:
                record.usage_status = 'normal'
            elif record.is_over_limit:
                record.usage_status = 'over_limit'
            elif record.usage_percentage >= 90:
                record.usage_status = 'critical'
            elif record.usage_percentage >= 80:
                record.usage_status = 'warning'
            else:
                record.usage_status = 'normal'

    def _search_usage_status(self, operator, value):
        """搜索使用状态"""
        # 由于 usage_status 是基于复杂计算的字段，我们简化搜索逻辑
        # 这里返回一个基本的域，实际应用中可能需要更复杂的逻辑
        if operator == '=' and value:
            return [('id', '>', 0)]  # 返回所有记录，让计算字段处理
        else:
            return [('id', '=', 0)]  # 返回空结果

    def _search_is_unlimited(self, operator, value):
        """搜索是否无限制"""
        # 简化搜索逻辑，返回基本域
        if operator == '=' and value:
            return [('id', '>', 0)]  # 返回所有记录，让计算字段处理
        else:
            return [('id', '>', 0)]  # 返回所有记录

    def _search_is_over_limit(self, operator, value):
        """搜索是否超限"""
        # 简化搜索逻辑，返回基本域
        if operator == '=' and value:
            return [('id', '>', 0)]  # 返回所有记录，让计算字段处理
        else:
            return [('id', '>', 0)]  # 返回所有记录

    def _search_usage_percentage(self, operator, value):
        """搜索使用百分比"""
        # 简化搜索逻辑，返回基本域
        return [('id', '>', 0)]  # 返回所有记录，让计算字段处理

    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            if record.is_unlimited:
                usage_info = f"{record.used_amount} / 无限制"
            else:
                usage_info = f"{record.used_amount} / {record.quota_limit}"
            
            name = f"{record.quota_definition_id.quota_name}: {usage_info}"
            
            # 添加状态标识
            status_dict = dict(record._fields['usage_status'].selection)
            status_name = status_dict.get(record.usage_status, '')
            if record.usage_status != 'normal':
                name += f" [{status_name}]"
            
            result.append((record.id, name))
        return result

    @api.model
    def increment_usage(self, organization_id, quota_code, amount=1):
        """增加额度使用量"""
        # 查找或创建使用记录
        organization = self.env['apollo.organization'].browse(organization_id)
        active_subscriptions = organization.subscription_ids.filtered('is_active_subscription')
        
        if not active_subscriptions:
            raise ValidationError(f'组织 {organization.name} 没有有效的订阅！')
        
        quota_definition = self.env['apollo.quota.definition'].search([
            ('quota_code', '=', quota_code)
        ], limit=1)
        
        if not quota_definition:
            raise ValidationError(f'额度定义 {quota_code} 不存在！')
        
        # 为每个有效订阅更新使用量
        for subscription in active_subscriptions:
            usage_record = self.search([
                ('organization_id', '=', organization_id),
                ('subscription_id', '=', subscription.id),
                ('quota_definition_id', '=', quota_definition.id)
            ], limit=1)
            
            if usage_record:
                usage_record.used_amount += amount
                usage_record.last_updated = fields.Datetime.now()
            else:
                self.create({
                    'organization_id': organization_id,
                    'subscription_id': subscription.id,
                    'quota_definition_id': quota_definition.id,
                    'used_amount': amount,
                })

    @api.model
    def reset_usage(self, organization_id, quota_code):
        """重置额度使用量"""
        quota_definition = self.env['apollo.quota.definition'].search([
            ('quota_code', '=', quota_code)
        ], limit=1)
        
        if quota_definition:
            usage_records = self.search([
                ('organization_id', '=', organization_id),
                ('quota_definition_id', '=', quota_definition.id)
            ])
            usage_records.write({
                'used_amount': 0,
                'last_updated': fields.Datetime.now()
            })

    @api.model
    def get_usage_summary(self, organization_id):
        """获取组织的额度使用汇总"""
        usage_records = self.search([('organization_id', '=', organization_id)])
        
        summary = {}
        for record in usage_records:
            quota_code = record.quota_definition_id.quota_code
            if quota_code not in summary:
                summary[quota_code] = {
                    'quota_name': record.quota_definition_id.quota_name,
                    'quota_category': record.quota_definition_id.quota_category,
                    'total_used': 0,
                    'quota_limit': record.quota_limit,
                    'is_unlimited': record.is_unlimited,
                }
            summary[quota_code]['total_used'] += record.used_amount
        
        return summary

    def action_reset_usage(self):
        """重置使用量"""
        for record in self:
            record.used_amount = 0
            record.last_updated = fields.Datetime.now()

    def action_adjust_usage(self):
        """调整使用量"""
        return {
            'name': '调整使用量',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.quota.usage.adjust.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {'default_usage_id': self.id},
        }
