# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError


class OrganizationSubscriptionExtension(models.Model):
    """组织套餐时长调整表 - 记录组织的套餐时长延长"""
    _name = 'apollo.organization.subscription.extension'
    _description = '组织套餐时长调整'
    _order = 'created_date desc, id'
    _rec_name = 'display_name'

    subscription_id = fields.Many2one(
        'apollo.organization.package.subscription',
        string='订阅',
        required=True,
        ondelete='cascade',
        help='关联的订阅'
    )
    extension_days = fields.Integer(
        string='延长天数',
        required=True,
        default=0,
        help='延长的天数'
    )
    original_end_date = fields.Date(
        string='原结束日期',
        required=True,
        help='延长前的结束日期'
    )
    new_end_date = fields.Date(
        string='新结束日期',
        required=True,
        help='延长后的结束日期'
    )
    reason = fields.Text(
        string='延长原因',
        help='延长的原因'
    )
    is_active = fields.Boolean(
        string='是否激活',
        default=True,
        help='是否激活此延长'
    )
    created_by = fields.Many2one(
        'res.users',
        string='创建人',
        required=True,
        default=lambda self: self.env.user,
        help='创建此延长的用户'
    )
    created_date = fields.Datetime(
        string='创建时间',
        default=fields.Datetime.now,
        readonly=True
    )

    # 关联字段 - 保留重要的外键关联用于查询性能
    organization_id = fields.Many2one(
        related='subscription_id.organization_id',
        string='组织',
        readonly=True,
        store=True
    )
    package_id = fields.Many2one(
        related='subscription_id.package_id',
        string='套餐',
        readonly=True,
        store=True
    )

    # 计算字段
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    extension_weeks = fields.Float(
        string='延长周数',
        compute='_compute_extension_weeks'
    )
    extension_description = fields.Char(
        string='延长描述',
        compute='_compute_extension_description'
    )

    _sql_constraints = [
        ('ext_days_positive', 'check(extension_days > 0)', 
         '延长天数必须大于0！'),
        ('new_end_date_check', 'check(new_end_date > original_end_date)', 
         '新结束日期必须晚于原结束日期！'),
    ]

    @api.depends('subscription_id', 'extension_days', 'new_end_date')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.subscription_id:
                org_name = record.organization_id.name if record.organization_id else '未知组织'
                pkg_name = record.package_id.name if record.package_id else '未知套餐'
                record.display_name = f"{org_name} - {pkg_name} (延长 {record.extension_days} 天)"
            else:
                record.display_name = "新延长"

    @api.depends('extension_days')
    def _compute_extension_weeks(self):
        """计算延长周数"""
        for record in self:
            record.extension_weeks = record.extension_days / 7.0

    @api.depends('extension_days', 'original_end_date', 'new_end_date')
    def _compute_extension_description(self):
        """计算延长描述"""
        for record in self:
            record.extension_description = f"从 {record.original_end_date} 延长 {record.extension_days} 天至 {record.new_end_date}"

    @api.constrains('original_end_date', 'new_end_date', 'extension_days')
    def _check_extension_consistency(self):
        """检查延长的一致性"""
        from datetime import timedelta
        for record in self:
            if record.original_end_date and record.extension_days:
                expected_new_date = record.original_end_date + timedelta(days=record.extension_days)
                if record.new_end_date != expected_new_date:
                    raise ValidationError('新结束日期与延长天数不匹配！')

    @api.onchange('original_end_date', 'extension_days')
    def _onchange_extension_calculation(self):
        """当原结束日期或延长天数改变时，自动计算新结束日期"""
        if self.original_end_date and self.extension_days:
            from datetime import timedelta
            self.new_end_date = self.original_end_date + timedelta(days=self.extension_days)

    @api.onchange('subscription_id')
    def _onchange_subscription_id(self):
        """当选择订阅时，自动设置原结束日期"""
        if self.subscription_id:
            # 获取当前的最终结束日期（考虑之前的延长）
            self.original_end_date = self.subscription_id.final_end_date

    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = f"延长 {record.extension_days} 天"
            if record.extension_weeks >= 1:
                name += f" ({record.extension_weeks:.1f} 周)"
            if not record.is_active:
                name += " [已禁用]"
            result.append((record.id, name))
        return result

    def action_toggle_active(self):
        """切换激活状态"""
        for record in self:
            record.is_active = not record.is_active

    @api.model_create_multi
    def create(self, vals_list):
        """创建时自动计算新结束日期"""
        for vals in vals_list:
            if vals.get('original_end_date') and vals.get('extension_days') and not vals.get('new_end_date'):
                from datetime import timedelta
                original_date = fields.Date.from_string(vals['original_end_date'])
                new_date = original_date + timedelta(days=vals['extension_days'])
                vals['new_end_date'] = fields.Date.to_string(new_date)
        return super().create(vals_list)

    @api.model
    def get_total_extension_days(self, subscription_id):
        """获取订阅的总延长天数"""
        extensions = self.search([
            ('subscription_id', '=', subscription_id),
            ('is_active', '=', True)
        ])
        return sum(extensions.mapped('extension_days'))

    @api.model
    def get_final_end_date(self, subscription_id):
        """获取订阅的最终结束日期"""
        subscription = self.env['apollo.organization.package.subscription'].browse(subscription_id)
        extensions = self.search([
            ('subscription_id', '=', subscription_id),
            ('is_active', '=', True)
        ])
        
        if extensions:
            return max(extensions.mapped('new_end_date'))
        else:
            return subscription.end_date
