# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError


class Package(models.Model):
    """主套餐表 - 定义所有套餐的基本信息"""
    _name = 'apollo.package'
    _description = '套餐'
    _order = 'sequence, created_date desc, id'
    _rec_name = 'name'

    name = fields.Char(
        string='套餐名称',
        required=True,
        size=100,
        help='套餐的显示名称'
    )
    package_code = fields.Char(
        string='套餐编码',
        required=True,
        size=50,
        help='套餐的唯一标识符'
    )

    
    duration_weeks = fields.Integer(
        string='套餐持续时间（周）',
        help='套餐的持续时间，为空表示永久'
    )
    is_active = fields.Boolean(
        string='是否激活',
        default=True,
        help='是否激活此套餐'
    )
    is_extension = fields.Boolean(
        string='是否为扩展套餐',
        default=False,
        help='是否为扩展套餐'
    )
    base_package_id = fields.Many2one(
        'apollo.package',
        string='基础套餐',
        help='扩展套餐的基础套餐'
    )
    description = fields.Text(
        string='套餐描述',
        help='套餐的详细描述'
    )
    sequence = fields.Integer(
        string='排序序号',
        default=10,
        help='用于排序的序号'
    )
    created_date = fields.Datetime(
        string='创建时间',
        default=fields.Datetime.now,
        readonly=True
    )
    updated_date = fields.Datetime(
        string='更新时间',
        default=fields.Datetime.now,
        readonly=True
    )

    # 关联字段
    package_quota_ids = fields.One2many(
        'apollo.package.quota',
        'package_id',
        string='套餐额度',
        domain=[('is_deleted', '=', False)]
    )
    deleted_quota_ids = fields.One2many(
        'apollo.package.quota',
        'package_id',
        string='已删除额度',
        domain=[('is_deleted', '=', True)],
        readonly=True
    )
    all_quota_ids = fields.One2many(
        'apollo.package.quota',
        'package_id',
        string='所有额度',
        help='包括已删除的额度，用于统计计算'
    )
    pricing_ids = fields.One2many(
        'apollo.package.pricing',
        'package_id',
        string='定价信息'
    )
    subscription_ids = fields.One2many(
        'apollo.organization.package.subscription',
        'package_id',
        string='订阅记录'
    )
    extension_package_ids = fields.One2many(
        'apollo.package',
        'base_package_id',
        string='扩展套餐'
    )

    # 计算字段
    current_price = fields.Float(
        string='当前价格',
        compute='_compute_current_price',
        help='当前有效的价格'
    )
    active_quota_count = fields.Integer(
        string='激活额度数量',
        compute='_compute_quota_counts'
    )
    total_quota_count = fields.Integer(
        string='总额度数量',
        compute='_compute_quota_counts'
    )
    deleted_quota_count = fields.Integer(
        string='已删除额度数量',
        compute='_compute_quota_counts'
    )
    subscription_count = fields.Integer(
        string='订阅数量',
        compute='_compute_subscription_count'
    )

    _sql_constraints = [
        ('package_code_unique', 'unique(package_code)', '套餐编码必须唯一！'),
    ]

    @api.constrains('is_extension', 'base_package_id')
    def _check_extension_package(self):
        """检查扩展套餐的基础套餐"""
        for record in self:
            if record.is_extension and not record.base_package_id:
                raise ValidationError('扩展套餐必须指定基础套餐！')
            if not record.is_extension and record.base_package_id:
                raise ValidationError('非扩展套餐不能指定基础套餐！')

    @api.constrains('base_package_id')
    def _check_recursive_base_package(self):
        """检查基础套餐不能形成循环引用"""
        for record in self:
            if record.base_package_id:
                current = record.base_package_id
                visited = {record.id}
                while current:
                    if current.id in visited:
                        raise ValidationError('基础套餐不能形成循环引用！')
                    visited.add(current.id)
                    current = current.base_package_id

    @api.depends('pricing_ids')
    def _compute_current_price(self):
        """计算当前有效价格"""
        for record in self:
            current_pricing = record.pricing_ids.filtered(
                lambda p: p.effective_date <= fields.Date.today() and 
                         (not p.expiry_date or p.expiry_date >= fields.Date.today())
            ).sorted('effective_date', reverse=True)
            record.current_price = current_pricing[0].price if current_pricing else 0.0

    @api.depends('all_quota_ids')
    def _compute_quota_counts(self):
        """计算额度数量"""
        for record in self:
            all_quotas = record.all_quota_ids
            record.active_quota_count = len(all_quotas.filtered(lambda q: q.is_active and not q.is_deleted))
            record.deleted_quota_count = len(all_quotas.filtered('is_deleted'))
            record.total_quota_count = len(all_quotas.filtered(lambda q: not q.is_deleted))

    @api.depends('subscription_ids')
    def _compute_subscription_count(self):
        """计算订阅数量"""
        for record in self:
            record.subscription_count = len(record.subscription_ids.filtered(
                lambda s: s.status == 'active'
            ))

    def write(self, vals):
        """更新时自动更新时间"""
        vals['updated_date'] = fields.Datetime.now()
        return super().write(vals)

    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = record.name
            if record.package_code:
                name = f"[{record.package_code}] {name}"
            if record.is_extension:
                name += " (扩展)"
            result.append((record.id, name))
        return result

    def action_view_quotas(self):
        """查看套餐额度"""
        return {
            'name': f'{self.name} - 额度设置',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.package.quota',
            'view_mode': 'list,form',
            'domain': [('package_id', '=', self.id)],
            'context': {'default_package_id': self.id},
        }

    def action_view_subscriptions(self):
        """查看订阅记录"""
        return {
            'name': f'{self.name} - 订阅记录',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.package.subscription',
            'view_mode': 'list,form',
            'domain': [('package_id', '=', self.id)],
            'context': {'default_package_id': self.id},
        }

    def action_quick_add_pricing(self):
        """快速添加套餐定价"""
        return {
            'type': 'ir.actions.act_window',
            'name': '快速添加套餐定价',
            'res_model': 'apollo.quick.add.pricing.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {'active_id': self.id}
        }
