# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError


class PackagePricing(models.Model):
    """套餐定价表 - 管理套餐的价格信息"""
    _name = 'apollo.package.pricing'
    _description = '套餐定价'
    _order = 'package_id, effective_date desc, id'
    _rec_name = 'display_name'

    package_id = fields.Many2one(
        'apollo.package',
        string='关联套餐',
        required=True,
        ondelete='cascade',
        help='关联的套餐'
    )
    price = fields.Float(
        string='价格',
        required=True,
        default=0.0,
        help='套餐价格'
    )
    currency = fields.Selection([
        ('CNY', '人民币'),
        ('USD', '美元'),
        ('EUR', '欧元'),
    ], string='货币类型', required=True, default='CNY', help='价格货币类型')
    

    effective_date = fields.Date(
        string='生效日期',
        required=True,
        default=fields.Date.today,
        help='价格生效日期'
    )
    expiry_date = fields.Date(
        string='失效日期',
        help='价格失效日期，为空表示永久有效'
    )



    # 计算字段
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    is_current = fields.Boolean(
        string='是否当前有效',
        compute='_compute_is_current',
        search='_search_is_current'
    )
    formatted_price = fields.Char(
        string='格式化价格',
        compute='_compute_formatted_price'
    )

    
    # 变更记录关联
    change_log_ids = fields.One2many(
        'apollo.package.pricing.change.log',
        'pricing_id',
        string='变更记录',
        readonly=True
    )
    change_log_count = fields.Integer(
        string='变更记录数量',
        compute='_compute_change_log_count'
    )

    _sql_constraints = [
        ('price_positive', 'check(price >= 0)', '价格不能为负数！'),
    ]

    @api.depends('package_id.name', 'price', 'currency', 'effective_date')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.package_id:
                currency_symbol = {'CNY': '¥', 'USD': '$', 'EUR': '€'}.get(record.currency, '')
                record.display_name = f"{record.package_id.name} - {currency_symbol}{record.price} ({record.effective_date})"
            else:
                record.display_name = "新定价"

    @api.depends('effective_date', 'expiry_date')
    def _compute_is_current(self):
        """计算是否当前有效"""
        today = fields.Date.today()
        for record in self:
            record.is_current = (
                record.effective_date <= today and 
                (not record.expiry_date or record.expiry_date >= today)
            )


    
    @api.depends('change_log_ids')
    def _compute_change_log_count(self):
        """计算变更记录数量"""
        for record in self:
            record.change_log_count = len(record.change_log_ids)
    
    @api.depends('price', 'currency')
    def _compute_formatted_price(self):
        """计算格式化价格"""
        for record in self:
            currency_symbol = {'CNY': '¥', 'USD': '$', 'EUR': '€'}.get(record.currency, '')
            record.formatted_price = f"{currency_symbol}{record.price:,.2f}"

    def _search_is_current(self, operator, value):
        """搜索当前有效的定价"""
        today = fields.Date.today()
        if operator == '=' and value:
            return [
                ('effective_date', '<=', today),
                '|',
                ('expiry_date', '=', False),
                ('expiry_date', '>=', today)
            ]
        elif operator == '=' and not value:
            return [
                '|',
                ('effective_date', '>', today),
                ('expiry_date', '<', today)
            ]
        else:
            return []

    @api.constrains('effective_date', 'expiry_date')
    def _check_date_range(self):
        """检查日期范围"""
        for record in self:
            if record.expiry_date and record.effective_date > record.expiry_date:
                raise ValidationError('生效日期不能晚于失效日期！')

    @api.constrains('package_id', 'effective_date', 'expiry_date')
    def _check_overlapping_pricing(self):
        """检查同一套餐的定价是否重叠"""
        for record in self:
            domain = [
                ('package_id', '=', record.package_id.id),
                ('id', '!=', record.id),
                ('effective_date', '<=', record.expiry_date or '9999-12-31'),
            ]
            if record.expiry_date:
                domain.append(('expiry_date', '>=', record.effective_date))
            else:
                domain.append('|')
                domain.append(('expiry_date', '>=', record.effective_date))
                domain.append(('expiry_date', '=', False))
            
            overlapping = self.search(domain)
            if overlapping:
                raise ValidationError(f'套餐 {record.package_id.name} 在此时间段内已有定价！')

    @api.model_create_multi
    def create(self, vals_list):
        """创建时记录变更"""
        # 确保vals_list是列表格式
        if not isinstance(vals_list, list):
            vals_list = [vals_list]
        
        # 临时移除只读限制以允许创建
        vals_list_copy = [dict(vals) for vals in vals_list]
        pricings = super().create(vals_list_copy)
        
        # 为每个创建的定价记录创建变更记录
        for pricing in pricings:
            self.env['apollo.package.pricing.change.log'].create_change_log(
                pricing_record=pricing,
                change_type='create',
                reason='新建定价'
            )
        
        return pricings
    
    def unlink(self):
        """删除时记录变更"""
        for pricing in self:
            # 创建删除记录
            self.env['apollo.package.pricing.change.log'].create_change_log(
                pricing_record=pricing,
                change_type='delete',
                reason='删除定价'
            )
        return super().unlink()
    
    def action_change_pricing(self):
        """打开定价修改向导"""
        return {
            'type': 'ir.actions.act_window',
            'name': '修改定价',
            'res_model': 'apollo.pricing.change.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {'active_id': self.id}
        }
    
    def action_view_change_logs(self):
        """查看变更记录"""
        return {
            'name': f'{self.display_name} - 变更记录',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.package.pricing.change.log',
            'view_mode': 'list,form',
            'domain': [('pricing_id', '=', self.id)],
            'context': {'default_pricing_id': self.id},
        }
    
    @api.model
    def create_pricing_with_log(self, vals, reason=None):
        """创建定价并记录变更（绕过只读限制）"""
        # 使用sudo()绕过权限限制
        pricing = self.sudo().create(vals)
        
        # 更新变更记录的原因
        if reason and pricing.change_log_ids:
            pricing.change_log_ids[-1].sudo().write({'reason': reason})
        
        return pricing
    
    def update_pricing_with_log(self, vals, reason=None):
        """更新定价并记录变更（使用sudo绕过权限限制）"""
        # 保存旧值
        old_values = {
            'price': self.price,
            'currency': self.currency,
            'effective_date': self.effective_date,
            'expiry_date': self.expiry_date,
        }
        
        # 使用sudo()绕过权限限制进行更新
        self.sudo().write(vals)
        
        # 创建变更记录
        self.env['apollo.package.pricing.change.log'].create_change_log(
            pricing_record=self,
            change_type='update',
            old_values=old_values,
            reason=reason or '通过修改向导更新'
        )
        
        return True
    
    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = record.formatted_price
            if record.is_current:
                name += " [当前]"
            result.append((record.id, name))
        return result
