# -*- coding: utf-8 -*-

from odoo import models, api
from odoo.exceptions import ValidationError
from datetime import datetime, date


class PackageQuotaService(models.AbstractModel):
    """套餐额度服务 - 提供额度计算和管理功能"""
    _name = 'apollo.package.quota.service'
    _description = '套餐额度服务'

    @api.model
    def get_organization_effective_quotas(self, organization_id, date_check=None):
        """
        获取组织有效额度（套餐额度 + 组织调整）
        
        Args:
            organization_id: 组织ID
            date_check: 检查日期，默认为今天
            
        Returns:
            dict: {quota_code: {'limit': int, 'is_unlimited': bool, 'source': str}}
        """
        if date_check is None:
            date_check = date.today()
        
        # 1. 获取组织套餐基础额度
        base_quotas = self._get_organization_package_quotas(organization_id, date_check)
        
        # 2. 获取组织额度调整
        adjustments = self._get_organization_quota_adjustments(organization_id, date_check)
        
        # 3. 计算最终额度
        effective_quotas = {}
        
        # 先处理基础额度
        for quota_code, quota_info in base_quotas.items():
            effective_quotas[quota_code] = {
                'limit': quota_info['limit'],
                'is_unlimited': quota_info['is_unlimited'],
                'source': 'package'
            }
        
        # 再应用调整
        for quota_code, adjustment_list in adjustments.items():
            if quota_code not in effective_quotas:
                # 如果基础额度中没有这个额度，创建一个默认的
                effective_quotas[quota_code] = {
                    'limit': 0,
                    'is_unlimited': False,
                    'source': 'adjustment'
                }
            
            # 按创建时间排序，依次应用调整
            for adj in sorted(adjustment_list, key=lambda x: x['created_date']):
                current_limit = effective_quotas[quota_code]['limit']
                
                if adj['adjustment_type'] == 'add':
                    effective_quotas[quota_code]['limit'] = current_limit + adj['adjustment_value']
                elif adj['adjustment_type'] == 'set':
                    effective_quotas[quota_code]['limit'] = adj['adjustment_value']
                elif adj['adjustment_type'] == 'multiply':
                    effective_quotas[quota_code]['limit'] = current_limit * adj['adjustment_value']
                
                effective_quotas[quota_code]['source'] = 'adjusted'
        
        return effective_quotas

    @api.model
    def _get_organization_package_quotas(self, organization_id, date_check):
        """获取组织套餐额度（只返回启用的额度）"""
        quotas = {}
        
        # 获取组织的有效订阅
        subscriptions = self.env['apollo.organization.package.subscription'].search([
            ('organization_id', '=', organization_id),
            ('status', '=', 'active'),
            ('start_date', '<=', date_check),
        ])
        
        for subscription in subscriptions:
            # 检查订阅是否真的有效（考虑延长）
            if not self._is_subscription_active(subscription, date_check):
                continue
            
            # 获取套餐额度
            package_quotas = self.env['apollo.package.quota'].search([
                ('package_id', '=', subscription.package_id.id),
                ('is_active', '=', True)
            ])
            
            for pq in package_quotas:
                quota_code = pq.quota_definition_id.quota_code
                
                if quota_code not in quotas:
                    quotas[quota_code] = {
                        'limit': 0,
                        'is_unlimited': False
                    }
                
                if pq.is_unlimited:
                    quotas[quota_code]['is_unlimited'] = True
                else:
                    if pq.is_additional:
                        # 追加额度
                        quotas[quota_code]['limit'] += pq.quota_limit
                    else:
                        # 基础额度，取最大值
                        quotas[quota_code]['limit'] = max(
                            quotas[quota_code]['limit'], 
                            pq.quota_limit
                        )
        
        return quotas

    @api.model
    def _get_organization_quota_adjustments(self, organization_id, date_check):
        """获取组织额度调整"""
        adjustments = {}
        
        adjustment_records = self.env['apollo.organization.quota.adjustment'].search([
            ('organization_id', '=', organization_id),
            ('is_active', '=', True),
            ('effective_date', '<=', date_check),
            '|',
            ('expiry_date', '=', False),
            ('expiry_date', '>=', date_check)
        ])
        
        for adj in adjustment_records:
            quota_code = adj.quota_definition_id.quota_code
            
            if quota_code not in adjustments:
                adjustments[quota_code] = []
            
            adjustments[quota_code].append({
                'adjustment_type': adj.adjustment_type,
                'adjustment_value': adj.adjustment_value,
                'created_date': adj.created_date,
            })
        
        return adjustments

    @api.model
    def _is_subscription_active(self, subscription, date_check):
        """检查订阅是否有效（考虑延长）"""
        # 获取最终结束日期
        final_end_date = subscription.end_date
        
        # 考虑延长
        extensions = self.env['apollo.organization.subscription.extension'].search([
            ('subscription_id', '=', subscription.id),
            ('is_active', '=', True)
        ])
        
        if extensions:
            final_end_date = max(extensions.mapped('new_end_date'))
        
        return date_check <= final_end_date

    @api.model
    def check_quota_available(self, organization_id, quota_code, required_amount=1):
        """
        检查组织是否有足够的额度使用某功能
        
        Args:
            organization_id: 组织ID
            quota_code: 额度编码
            required_amount: 需要的额度数量
            
        Returns:
            dict: {'available': bool, 'current_usage': int, 'quota_limit': int, 'remaining': int}
        """
        # 获取有效额度
        effective_quotas = self.get_organization_effective_quotas(organization_id)
        
        if quota_code not in effective_quotas:
            return {
                'available': False,
                'current_usage': 0,
                'quota_limit': 0,
                'remaining': 0,
                'is_unlimited': False,
                'message': f'组织没有额度 {quota_code}'
            }
        
        quota_info = effective_quotas[quota_code]
        
        # 如果是无限制额度
        if quota_info['is_unlimited']:
            return {
                'available': True,
                'current_usage': 0,
                'quota_limit': -1,  # -1 表示无限制
                'remaining': -1,
                'is_unlimited': True,
                'message': '无限制额度'
            }
        
        # 获取当前使用量
        current_usage = self._get_organization_quota_usage(organization_id, quota_code)
        quota_limit = quota_info['limit']
        remaining = quota_limit - current_usage
        
        return {
            'available': remaining >= required_amount,
            'current_usage': current_usage,
            'quota_limit': quota_limit,
            'remaining': remaining,
            'is_unlimited': False,
            'message': f'剩余 {remaining}/{quota_limit}'
        }

    @api.model
    def _get_organization_quota_usage(self, organization_id, quota_code):
        """获取组织额度使用量"""
        quota_definition = self.env['apollo.quota.definition'].search([
            ('quota_code', '=', quota_code)
        ], limit=1)
        
        if not quota_definition:
            return 0
        
        usage_records = self.env['apollo.organization.quota.usage'].search([
            ('organization_id', '=', organization_id),
            ('quota_definition_id', '=', quota_definition.id)
        ])
        
        return sum(usage_records.mapped('used_amount'))

    @api.model
    def consume_quota(self, organization_id, quota_code, amount=1):
        """
        消费额度
        
        Args:
            organization_id: 组织ID
            quota_code: 额度编码
            amount: 消费数量
            
        Returns:
            dict: {'success': bool, 'message': str}
        """
        # 检查额度是否可用
        quota_check = self.check_quota_available(organization_id, quota_code, amount)
        
        if not quota_check['available']:
            return {
                'success': False,
                'message': f'额度不足：{quota_check["message"]}'
            }
        
        # 增加使用量
        try:
            self.env['apollo.organization.quota.usage'].increment_usage(
                organization_id, quota_code, amount
            )
            return {
                'success': True,
                'message': f'成功消费额度 {quota_code}: {amount}'
            }
        except Exception as e:
            return {
                'success': False,
                'message': f'消费额度失败：{str(e)}'
            }

    @api.model
    def get_organization_quota_summary(self, organization_id):
        """
        获取组织额度汇总信息
        
        Args:
            organization_id: 组织ID
            
        Returns:
            dict: 额度汇总信息
        """
        effective_quotas = self.get_organization_effective_quotas(organization_id)
        usage_summary = self.env['apollo.organization.quota.usage'].get_usage_summary(organization_id)
        
        summary = {}
        
        # 合并额度信息和使用情况
        all_quota_codes = set(effective_quotas.keys()) | set(usage_summary.keys())
        
        for quota_code in all_quota_codes:
            quota_info = effective_quotas.get(quota_code, {'limit': 0, 'is_unlimited': False})
            usage_info = usage_summary.get(quota_code, {'total_used': 0, 'quota_name': quota_code})
            
            summary[quota_code] = {
                'quota_name': usage_info.get('quota_name', quota_code),
                'quota_category': usage_info.get('quota_category', 'unknown'),
                'quota_limit': quota_info['limit'],
                'is_unlimited': quota_info['is_unlimited'],
                'used_amount': usage_info['total_used'],
                'remaining_amount': quota_info['limit'] - usage_info['total_used'] if not quota_info['is_unlimited'] else -1,
                'usage_percentage': (usage_info['total_used'] / quota_info['limit'] * 100) if quota_info['limit'] > 0 else 0,
                'source': quota_info.get('source', 'unknown')
            }
        
        return summary

    @api.model
    def validate_package_quotas(self, package_id):
        """
        验证套餐额度配置的合理性
        
        Args:
            package_id: 套餐ID
            
        Returns:
            dict: {'valid': bool, 'warnings': list, 'errors': list}
        """
        package = self.env['apollo.package'].browse(package_id)
        warnings = []
        errors = []
        
        if not package.exists():
            errors.append('套餐不存在')
            return {'valid': False, 'warnings': warnings, 'errors': errors}
        
        # 检查是否有额度
        if not package.package_quota_ids:
            warnings.append('套餐没有配置任何额度')
        
        # 检查功能权限额度
        feature_quotas = package.package_quota_ids.filtered(
            lambda q: q.quota_definition_id.quota_category == 'features'
        )
        
        for quota in feature_quotas:
            if quota.is_active and quota.quota_limit <= 0 and not quota.is_unlimited:
                warnings.append(f'功能额度 {quota.quota_name} 的限制为0，用户将无法使用此功能')
        
        # 检查扩展套餐
        if package.is_extension:
            if not package.base_package_id:
                errors.append('扩展套餐必须指定基础套餐')
            else:
                # 检查扩展额度是否合理
                base_quotas = package.base_package_id.package_quota_ids
                extension_quotas = package.package_quota_ids
                
                for ext_quota in extension_quotas:
                    if not ext_quota.is_additional:
                        warnings.append(f'扩展套餐中的额度 {ext_quota.quota_name} 建议设置为追加额度')
        
        # 检查定价
        if not package.pricing_ids:
            warnings.append('套餐没有配置价格信息')
        
        return {
            'valid': len(errors) == 0,
            'warnings': warnings,
            'errors': errors
        }
