# -*- coding: utf-8 -*-

import logging
from odoo import _
from odoo.http import request, route
from ..base_api_controller import BaseApiController

_logger = logging.getLogger(__name__)


class AuthApiController(BaseApiController):
    """用户认证相关API控制器"""
    
    @route('/api/apollo/jwt/verify', type='json', auth='public', methods=['POST'], csrf=False)
    def verify_jwt_token(self):
        """验证JWT令牌"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['token'])
            if error:
                return error
            
            token = data['token']
            
            # 验证JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            verify_result = jwt_service.verify_token(token)
            
            if not verify_result['valid']:
                return self._error_response('令牌无效或已过期', 401)
            
            # 构建响应数据
            response_data = {
                'valid': True,
                'phone': verify_result['phone'],
                'status': verify_result['status'],
                'status_description': verify_result['status_description'],
                'expires_at': verify_result['expires_at']
            }
            
            # 如果是已登录状态，添加用户信息
            if verify_result['status'] == 'login_success' and verify_result.get('user_id'):
                user = request.env['res.users'].sudo().browse(verify_result['user_id'])
                if user.exists():
                    response_data.update({
                        'user_id': user.id,
                        'user_name': user.name
                    })
            
            return self._success_response('令牌验证成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "验证JWT令牌")
    
    @route('/api/apollo/user/register', type='json', auth='public', methods=['POST'], csrf=False)
    def register_user(self):
        """完成用户注册"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['token', 'name', 'password']
            )
            if error:
                return error
            
            token = data['token']
            name = data['name']
            password = data['password']
            
            # 验证JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            verify_result = jwt_service.verify_token(token)
            
            if not verify_result['valid']:
                return self._error_response('令牌无效或已过期', 401)
            
            # 检查令牌状态是否为待注册
            if verify_result['status'] != 'registration_pending':
                return self._error_response('令牌状态不正确，无法注册', 400)
            
            phone = verify_result['phone']
            organization_id = verify_result.get('organization_id')
            invitation_code = verify_result.get('invitation_code')
            
            # 确保token中包含organization_id
            if not organization_id:
                return self._error_response('令牌中缺少组织信息', 400)
            
            # 验证组织是否存在
            organization = request.env['apollo.organization'].sudo().search([
                ('id', '=', organization_id),
                ('active', '=', True)
            ], limit=1)
            
            if not organization:
                return self._error_response('组织不存在或已停用', 404)
            
            # 检查用户是否已存在
            existing_user = request.env['res.users'].sudo().search([
                ('mobile', '=', phone)
            ], limit=1)
            
            if existing_user:
                return self._error_response('该手机号已注册', 400)
            
            # 创建用户
            user_data = {
                'name': name,
                'login': phone,  # 使用手机号作为登录名
                'mobile': phone,
                'password': password,  # 设置用户密码
                'active': True,
                'groups_id': [(6, 0, [request.env.ref('base.group_user').id])]
            }
            
            user = request.env['res.users'].sudo().create(user_data)
            
            # 查找并绑定邀请码
            search_domain = [
                ('phone', '=', phone),
                ('organization_id', '=', organization_id),
                ('active', '=', True),
                ('voided', '=', False),
                ('bound_user_id', '=', False)
            ]
            
            # 如果令牌中包含邀请码，优先使用该邀请码
            if invitation_code:
                search_domain.append(('code', '=', invitation_code))
            
            invitation = request.env['apollo.invitation.code'].sudo().search(search_domain, limit=1)
            
            if invitation:
                # 绑定邀请码到用户（这会自动创建组织成员记录并设置owner）
                invitation.bind_user(user.id)
                _logger.info(f"用户注册成功并绑定邀请码: {user.name} -> {invitation.code}")
            else:
                # 如果没有找到邀请码，直接创建组织成员记录
                request.env['apollo.organization.member'].sudo().create({
                    'user_id': user.id,
                    'organization_id': organization_id,
                    'join_date': request.env.cr.now()
                })
                _logger.warning(f"用户注册时未找到对应的邀请码: {phone}, 组织: {organization_id}")
            
            # 生成新的登录令牌
            login_token_data = jwt_service.generate_login_token(
                user_id=user.id,
                phone=phone
            )
            
            # 构建响应数据
            response_data = {
                'phone': phone,
                'status': 'login_success',
                'status_description': '注册成功，已登录',
                'token': login_token_data['token'],
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile
                }
            }
            
            return self._success_response('注册成功', response_data, 201)
            
        except Exception as e:
            return self._handle_exception(e, "用户注册")
    
    @route('/api/apollo/user/login', type='json', auth='public', methods=['POST'], csrf=False)
    def login_user(self):
        """通过手机号或邮箱和密码登录"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['password'],
                optional_fields=['phone', 'email']
            )
            if error:
                return error
            
            phone = data.get('phone')
            email = data.get('email')
            password = data['password']
            
            # 确保提供了手机号或邮箱中的至少一个
            if not phone and not email:
                return self._error_response('请提供手机号或邮箱', 400)
            
            # 查找用户
            search_domain = [('active', '=', True)]
            
            # 构建搜索条件：支持手机号或邮箱查找
            if phone and email:
                # 同时提供了手机号和邮箱，使用OR条件
                search_domain.append('|')
                search_domain.append(('mobile', '=', phone))
                search_domain.append(('email', '=', email))
                _logger.info(f"用户登录：同时使用手机号 {phone} 和邮箱 {email} 查找用户")
            elif phone:
                # 只提供了手机号
                search_domain.append(('mobile', '=', phone))
                _logger.info(f"用户登录：使用手机号 {phone} 查找用户")
            elif email:
                # 只提供了邮箱
                search_domain.append(('email', '=', email))
                _logger.info(f"用户登录：使用邮箱 {email} 查找用户")
            
            _logger.info(f"用户查找条件: {search_domain}")
            user = request.env['res.users'].sudo().search(search_domain, limit=1)
            
            if not user:
                _logger.warning(f"未找到匹配的用户，查找条件: {search_domain}")
                # 额外检查：查看是否有类似的用户记录
                if phone:
                    similar_users = request.env['res.users'].sudo().search([
                        ('mobile', 'ilike', phone)
                    ], limit=5)
                    _logger.info(f"类似手机号的用户: {[(u.login, u.mobile, u.active) for u in similar_users]}")
                if email:
                    similar_users = request.env['res.users'].sudo().search([
                        ('email', 'ilike', email)
                    ], limit=5)
                    _logger.info(f"类似邮箱的用户: {[(u.login, u.email, u.active) for u in similar_users]}")
                return self._error_response('用户不存在或已停用', 404)
            
            _logger.info(f"找到用户: {user.login}, mobile: {user.mobile}, email: {user.email}, active: {user.active}")
            
            # 验证密码
            try:
                # 使用 Odoo 的认证机制验证密码
                from odoo.modules.registry import Registry
                from odoo.exceptions import AccessDenied
                
                credential = {
                    'login': user.login,
                    'password': password,
                    'type': 'password',
                }
                
                _logger.info(f"开始验证用户密码: login={user.login}")
                
                # 使用正确的认证方法
                registry = Registry(request.env.cr.dbname)
                auth_result = registry['res.users'].authenticate(
                    request.env.cr.dbname, 
                    credential, 
                    {'interactive': False}
                )
                
                _logger.info(f"认证结果: {auth_result}")
                
                if not auth_result or auth_result.get('uid') != user.id:
                    _logger.warning(f"密码验证失败: 期望uid={user.id}, 实际认证结果={auth_result}")
                    return self._error_response('密码错误', 401)
                    
                _logger.info(f"用户 {user.login} 密码验证成功")
                    
            except AccessDenied as e:
                _logger.warning(f"认证被拒绝: {e}")
                return self._error_response('密码错误', 401)
            except Exception as e:
                _logger.error(f"认证过程出错: {e}")
                return self._error_response('认证失败，请稍后重试', 500)
            
            # 生成登录令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            login_token_data = jwt_service.generate_login_token(
                user_id=user.id,
                phone=user.mobile  # 使用用户实际的手机号
            )
            
            # 获取用户的组织信息
            member_records = request.env['apollo.organization.member'].sudo().search([
                ('user_id', '=', user.id),
                ('leave_date', '=', False)  # 只获取当前成员身份
            ])
            
            organizations = []
            for member in member_records:
                organizations.append({
                    'id': member.organization_id.id,
                    'name': member.organization_id.name,
                    'join_date': member.join_date.isoformat() if member.join_date else None,
                    'is_owner': member.is_owner,
                    'role': 'owner' if member.is_owner else 'member'
                })
            
            # 构建响应数据
            response_data = {
                'phone': user.mobile,  # 使用用户实际的手机号
                'status': 'login_success',
                'status_description': '登录成功',
                'token': login_token_data['token'],
                'expires_at': login_token_data.get('expires_at'),
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile
                },
                'organizations': organizations
            }
            
            return self._success_response('登录成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "用户登录")
    
    @route('/api/apollo/user/login/sms', type='json', auth='public', methods=['POST'], csrf=False)
    def login_with_sms_code(self):
        """通过手机号和短信验证码登录"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['phone', 'sms_code']
            )
            if error:
                return error
            
            phone = data['phone']
            sms_verification_code = data['sms_code']
            
            # 验证短信验证码
            sms_model = request.env['apollo.sms.verification.code'].sudo()
            is_valid, message = sms_model.verify_login_code(phone, sms_verification_code)
            
            if not is_valid:
                return self._error_response(message, 400)
            
            # 查找用户
            user = request.env['res.users'].sudo().search([
                ('mobile', '=', phone),
                ('active', '=', True)
            ], limit=1)
            
            if not user:
                return self._error_response('用户不存在或已停用', 404)
            
            # 生成登录令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            login_token_data = jwt_service.generate_login_token(
                user_id=user.id,
                phone=phone
            )
            
            # 获取用户的组织信息
            member_records = request.env['apollo.organization.member'].sudo().search([
                ('user_id', '=', user.id),
                ('leave_date', '=', False)  # 只获取当前成员身份
            ])
            
            organizations = []
            for member in member_records:
                organizations.append({
                    'id': member.organization_id.id,
                    'name': member.organization_id.name,
                    'join_date': member.join_date.isoformat() if member.join_date else None,
                    'is_owner': member.is_owner,
                    'role': 'owner' if member.is_owner else 'member'
                })
            
            # 构建响应数据
            response_data = {
                'phone': phone,
                'status': 'login_success',
                'status_description': '登录成功',
                'token': login_token_data['token'],
                'expires_at': login_token_data.get('expires_at'),
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile
                },
                'organizations': organizations
            }
            
            return self._success_response('登录成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "短信验证码登录")
    
    @route('/api/apollo/user/profile', type='json', auth='public', methods=['POST'], csrf=False)
    def get_user_profile(self):
        """获取用户个人资料（基本信息）"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['token'])
            if error:
                return error
            
            token = data['token']
            
            # 验证JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            verify_result = jwt_service.verify_token(token)
            
            if not verify_result['valid']:
                return self._error_response('令牌无效或已过期', 401)
            
            # 检查是否为已登录状态
            if verify_result['status'] != 'login_success':
                return self._error_response('用户未登录', 401)
            
            user_id = verify_result.get('user_id')
            if not user_id:
                return self._error_response('令牌中缺少用户信息', 400)
            
            # 获取用户信息
            user = request.env['res.users'].sudo().browse(user_id)
            if not user.exists():
                return self._error_response('用户不存在', 404)
            
            # 构建用户基本信息响应数据
            response_data = {
                'id': user.id,
                'name': user.name,
                'email': user.email or '',
                'mobile': user.mobile or '',
                'active': user.active,
                'login': user.login,
                'create_date': user.create_date.isoformat() if user.create_date else None,
                'last_login': user.login_date.isoformat() if user.login_date else None
            }
            
            return self._success_response('获取用户资料成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "获取用户资料")
    
    @route('/api/apollo/user/organizations', type='json', auth='public', methods=['POST'], csrf=False)
    def get_user_organizations(self):
        """获取用户组织资料（组织成员信息）"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['token'])
            if error:
                return error
            
            token = data['token']
            
            # 验证JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            verify_result = jwt_service.verify_token(token)
            
            if not verify_result['valid']:
                return self._error_response('令牌无效或已过期', 401)
            
            # 检查是否为已登录状态
            if verify_result['status'] != 'login_success':
                return self._error_response('用户未登录', 401)
            
            user_id = verify_result.get('user_id')
            if not user_id:
                return self._error_response('令牌中缺少用户信息', 400)
            
            # 获取用户信息
            user = request.env['res.users'].sudo().browse(user_id)
            if not user.exists():
                return self._error_response('用户不存在', 404)
            
            # 获取用户的组织成员信息
            member_records = request.env['apollo.organization.member'].sudo().search([
                ('user_id', '=', user.id),
                ('leave_date', '=', False)  # 只获取当前成员身份
            ])
            
            organizations = []
            for member in member_records:
                org_data = {
                    'id': member.organization_id.id,
                    'name': member.organization_id.name,
                    'logo': member.organization_id.logo,
                    'active': member.organization_id.active,
                    'member_info': {
                        'join_date': member.join_date.isoformat() if member.join_date else None,
                        'is_owner': member.is_owner,
                        'role': 'owner' if member.is_owner else 'member',
                        'is_active': member.is_active,
                        'added_by': member.added_by_id.name if member.added_by_id else None,
                        'note': member.note or ''
                    }
                }
                
                # 如果是组织所有者，添加组织统计信息
                if member.is_owner:
                    org_data['organization_stats'] = {
                        'total_members': len(member.organization_id.active_member_ids),
                        'invitation_codes_count': member.organization_id.invitation_code_count,
                        'owner_change_count': member.organization_id.owner_change_count
                    }
                
                organizations.append(org_data)
            
            # 构建响应数据
            response_data = {
                'organizations': organizations,
                'total_count': len(organizations),
                'has_owner_role': any(org['member_info']['is_owner'] for org in organizations)
            }
            
            return self._success_response('获取用户组织资料成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "获取用户组织资料")
