# -*- coding: utf-8 -*-

import logging
from odoo import _
from odoo.http import request, route
from ..base_api_controller import BaseApiController

_logger = logging.getLogger(__name__)


class InvitationApiController(BaseApiController):
    """邀请码相关API控制器"""
    
    @route('/api/apollo/invitation/info', type='json', auth='public', methods=['POST'], csrf=False)
    def get_invitation_code_info(self):
        """获取邀请码信息"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['invitation_code'])
            if error:
                return error
            
            invitation_code = data['invitation_code']
            
            # 查找邀请码
            invitation = self._find_invitation_by_code(invitation_code)
            if not invitation:
                return self._error_response('邀请码不存在或已失效', 404)
            
            # 构建响应数据
            response_data = {
                'invitation_code': invitation.code,
                'phone': invitation.phone,
                'organization': {
                    'id': invitation.organization_id.id,
                    'name': invitation.organization_id.name,
                    'logo': invitation.organization_id.logo or None
                },
                'voided': invitation.voided,
                'void_date': invitation.void_date.isoformat() if invitation.void_date else None,
                'create_date': invitation.create_date.isoformat() if invitation.create_date else None,
                'bound_user_id': invitation.bound_user_id.id if invitation.bound_user_id else None,
                'bound_user_name': invitation.bound_user_id.name if invitation.bound_user_id else None,
                'bound_date': invitation.bound_date.isoformat() if invitation.bound_date else None
            }
            
            return self._success_response('获取成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "获取邀请码信息")
    
    @route('/api/apollo/invitation/create', type='json', auth='bearer', methods=['POST'], csrf=False)
    def create_invitation_code(self):
        """创建邀请码"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['organization_id', 'phone']
            )
            if error:
                return error
            
            organization_id = data['organization_id']
            phone = data['phone']
            
            # 验证手机号
            if not self._validate_phone_number(phone):
                return self._error_response('手机号格式不正确')
            
            # 验证组织是否存在
            organization = request.env['apollo.organization'].search([
                ('id', '=', organization_id),
                ('active', '=', True)
            ], limit=1)
            
            if not organization:
                return self._error_response('组织不存在或已停用', 404)
            
            # 检查是否已存在相同手机号和组织的邀请码
            existing_invitation = request.env['apollo.invitation.code'].search([
                ('phone', '=', phone),
                ('organization_id', '=', organization_id),
                ('active', '=', True),
                ('voided', '=', False)
            ], limit=1)
            
            if existing_invitation:
                return self._error_response('该手机号已有有效的邀请码')
            
            # 创建邀请码
            invitation = request.env['apollo.invitation.code'].create({
                'phone': phone,
                'organization_id': organization_id,
                'created_by': request.env.user.id
            })
            
            # 构建响应数据
            response_data = {
                'id': invitation.id,
                'invitation_code': invitation.code,
                'phone': invitation.phone,
                'organization': {
                    'id': organization.id,
                    'name': organization.name
                },
                'used': False,
                'create_date': invitation.create_date.isoformat()
            }
            
            return self._success_response('邀请码创建成功', response_data, 201)
            
        except Exception as e:
            return self._handle_exception(e, "创建邀请码")
    
    @route('/api/apollo/invitation/void', type='json', auth='bearer', methods=['POST'], csrf=False)
    def void_invitation_code(self):
        """作废邀请码"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['invitation_code'])
            if error:
                return error
            
            invitation_code = data['invitation_code']
            
            # 查找邀请码
            invitation = self._find_invitation_by_code(invitation_code)
            if not invitation:
                return self._error_response('邀请码不存在或已失效', 404)
            
            # 检查是否已作废
            if invitation.voided:
                return self._error_response('邀请码已经作废')
            
            # 作废邀请码
            invitation.void_code()
            
            return self._success_response('邀请码已作废')
            
        except Exception as e:
            return self._handle_exception(e, "作废邀请码")
    
    def _find_invitation_by_code(self, invitation_code):
        """根据邀请码查找记录
        
        Args:
            invitation_code (str): 邀请码
            
        Returns:
            recordset: 邀请码记录或空记录集
        """
        return request.env['apollo.invitation.code'].sudo().search([
            ('code', '=', invitation_code),
            ('active', '=', True),
            ('voided', '=', False)
        ], limit=1)
