# -*- coding: utf-8 -*-

import logging
from datetime import datetime, timedelta
from odoo import fields, _
from odoo.http import request, route
from ..base_api_controller import BaseApiController

_logger = logging.getLogger(__name__)


class SmsApiController(BaseApiController):
    """短信验证码相关API控制器"""
    
    @route('/api/apollo/sms/send_invitation_code', type='json', auth='public', methods=['POST'], csrf=False)
    def send_invitation_sms_code(self):
        """发送邀请注册短信验证码"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['invitation_code'])
            if error:
                return error
            
            invitation_code = data['invitation_code']
            
            # 查找邀请码
            invitation = self._find_invitation_by_code(invitation_code)
            if not invitation:
                return self._error_response('邀请码不存在或已失效', 404)
            
            if invitation.voided:
                return self._error_response('邀请码已作废', 400)
            
            phone = invitation.phone
            organization_id = invitation.organization_id.id
            
            # 检查是否在冷却期内
            cooldown_error = self._check_sms_cooldown(phone)
            if cooldown_error:
                return cooldown_error
            
            # 获取客户端信息
            client_info = self._get_client_info()
            
            # 创建验证码记录
            sms_service = request.env['apollo.sms.service'].sudo()
            verification_code = sms_service.create_verification_code(
                phone=phone,
                invitation_code=invitation_code,
                organization_id=organization_id,
                ip_address=client_info['ip_address'],
                user_agent=client_info['user_agent']
            )
            
            # 发送短信
            send_result = sms_service.send_verification_sms(verification_code)
            
            if send_result['success']:
                # 构建响应数据
                response_data = {
                    'phone': phone,
                    'masked_phone': self._mask_phone_number(phone),
                    'expires_in': 300,  # 5分钟
                    'message': '验证码已发送'
                }
                return self._success_response('验证码发送成功', response_data)
            else:
                return self._error_response(f'验证码发送失败: {send_result["message"]}', 500)
                
        except Exception as e:
            return self._handle_exception(e, "发送验证码")
    
    @route('/api/apollo/sms/verify_invitation_code', type='json', auth='public', methods=['POST'], csrf=False)
    def verify_invitation_sms_code(self):
        """验证邀请注册短信验证码"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['invitation_code', 'sms_code']
            )
            if error:
                return error
            
            invitation_code = data['invitation_code']
            sms_verification_code = data['sms_code']
            
            # 查找邀请码
            invitation = self._find_invitation_by_code(invitation_code)
            if not invitation:
                return self._error_response('邀请码不存在或已失效', 404)
            
            if invitation.voided:
                return self._error_response('邀请码已作废', 400)
            
            phone = invitation.phone
            organization = invitation.organization_id
            
            # 验证短信验证码
            sms_service = request.env['apollo.sms.service'].sudo()
            verify_result = sms_service.verify_sms_code(
                phone=phone,
                code=sms_verification_code,
                invitation_code=invitation_code
            )
            
            if not verify_result['success']:
                return self._error_response(verify_result['message'], 400)
            
            # 检查用户是否已存在
            user = request.env['res.users'].sudo().search([
                ('mobile', '=', phone)
            ], limit=1)
            
            # 构建基础响应数据
            response_data = {
                'phone': phone,
                'organization': {
                    'id': organization.id,
                    'name': organization.name
                },
                'verified': True
            }
            
            # 生成JWT令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            
            if user:
                # 用户已存在，检查是否需要绑定邀请码
                invitation = self._find_invitation_by_code(invitation_code)
                if invitation and not invitation.bound_user_id:
                    try:
                        invitation.bind_user(user.id)
                        _logger.info(f"已存在用户绑定邀请码成功: {user.name} -> {invitation.code}")
                    except Exception as e:
                        _logger.warning(f"已存在用户绑定邀请码失败: {user.name} -> {invitation.code}, 错误: {str(e)}")
                
                # 生成登录令牌
                token_data = jwt_service.generate_login_token(
                    user_id=user.id,
                    phone=phone
                )
                
                # 获取用户的组织信息
                member_records = request.env['apollo.organization.member'].sudo().search([
                    ('user_id', '=', user.id),
                    ('leave_date', '=', False)  # 只获取当前成员身份
                ])
                
                organizations = []
                for member in member_records:
                    organizations.append({
                        'id': member.organization_id.id,
                        'name': member.organization_id.name,
                        'join_date': member.join_date.isoformat() if member.join_date else None,
                        'is_owner': member.is_owner,
                        'role': 'owner' if member.is_owner else 'member'
                    })
                
                response_data.update({
                    'user_exists': True,
                    'status': 'login_success',
                    'status_description': '验证成功，已自动登录',
                    'token': token_data['token'],
                    'expires_at': token_data.get('expires_at'),
                    'user': {
                        'id': user.id,
                        'name': user.name,
                        'email': user.email or '',
                        'mobile': user.mobile or phone
                    },
                    'organizations': organizations
                })
                
                return self._success_response('验证成功，已存在用户自动登录', response_data)
            else:
                # 用户不存在，需要注册
                token_data = jwt_service.generate_registration_token(
                    phone=phone,
                    organization_id=organization.id,
                    invitation_code=invitation_code
                )
                
                response_data.update({
                    'user_exists': False,
                    'status': 'registration_pending',
                    'status_description': '待注册',
                    'token': token_data['token']
                })
                
                return self._success_response('验证成功，需要完成注册', response_data)
                
        except Exception as e:
            return self._handle_exception(e, "验证短信验证码")
    
    @route('/api/apollo/sms/send_login_code', type='json', auth='public', methods=['POST'], csrf=False)
    def send_login_sms_code(self):
        """发送登录短信验证码"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(required_fields=['phone'])
            if error:
                return error
            
            phone = data['phone']
            
            # 验证手机号格式
            if not self._validate_phone_number(phone):
                return self._error_response('手机号格式不正确')
            
            # 检查用户是否存在
            user = request.env['res.users'].sudo().search([
                ('mobile', '=', phone),
                ('active', '=', True)
            ], limit=1)
            
            if not user:
                return self._error_response('该手机号未注册', 404)
            
            # 检查是否在冷却期内
            cooldown_error = self._check_sms_cooldown(phone)
            if cooldown_error:
                return cooldown_error
            
            # 获取客户端信息
            client_info = self._get_client_info()
            
            # 创建登录验证码记录
            sms_model = request.env['apollo.sms.verification.code'].sudo()
            verification_record = sms_model.create_login_verification_code(
                phone=phone,
                ip_address=client_info['ip_address'],
                user_agent=client_info['user_agent']
            )
            
            # 发送短信
            sms_service = request.env['apollo.sms.service'].sudo()
            send_result = sms_service.send_verification_sms(verification_record)
            
            if send_result['success']:
                # 构建响应数据
                response_data = {
                    'phone': phone,
                    'masked_phone': self._mask_phone_number(phone),
                    'expires_in': 300,  # 5分钟
                    'message': '登录验证码已发送'
                }
                return self._success_response('登录验证码发送成功', response_data)
            else:
                return self._error_response(f'登录验证码发送失败: {send_result["message"]}', 500)
                
        except Exception as e:
            return self._handle_exception(e, "发送登录验证码")
    
    @route('/api/apollo/sms/verify_login_code', type='json', auth='public', methods=['POST'], csrf=False)
    def verify_login_sms_code(self):
        """验证登录短信验证码"""
        try:
            # 验证请求数据
            data, error = self._validate_json_data(
                required_fields=['phone', 'sms_code']
            )
            if error:
                return error
            
            phone = data['phone']
            sms_verification_code = data['sms_code']
            
            # 验证短信验证码
            sms_model = request.env['apollo.sms.verification.code'].sudo()
            is_valid, message = sms_model.verify_login_code(phone, sms_verification_code)
            
            if not is_valid:
                return self._error_response(message, 400)
            
            # 查找用户
            user = request.env['res.users'].sudo().search([
                ('mobile', '=', phone),
                ('active', '=', True)
            ], limit=1)
            
            if not user:
                return self._error_response('用户不存在或已停用', 404)
            
            # 生成登录令牌
            jwt_service = request.env['apollo.jwt.service'].sudo()
            login_token_data = jwt_service.generate_login_token(
                user_id=user.id,
                phone=phone
            )
            
            # 获取用户的组织信息
            member_records = request.env['apollo.organization.member'].sudo().search([
                ('user_id', '=', user.id),
                ('leave_date', '=', False)  # 只获取当前成员身份
            ])
            
            organizations = []
            for member in member_records:
                organizations.append({
                    'id': member.organization_id.id,
                    'name': member.organization_id.name,
                    'join_date': member.join_date.isoformat() if member.join_date else None
                })
            
            # 构建响应数据
            response_data = {
                'phone': phone,
                'status': 'login_success',
                'status_description': '登录成功',
                'token': login_token_data['token'],
                'expires_at': login_token_data.get('expires_at'),
                'user': {
                    'id': user.id,
                    'name': user.name,
                    'email': user.email or '',
                    'mobile': user.mobile
                },
                'organizations': organizations
            }
            
            return self._success_response('登录成功', response_data)
            
        except Exception as e:
            return self._handle_exception(e, "验证登录验证码")
    
    def _find_invitation_by_code(self, invitation_code):
        """根据邀请码查找记录"""
        return request.env['apollo.invitation.code'].sudo().search([
            ('code', '=', invitation_code),
            ('active', '=', True),
            ('voided', '=', False)
        ], limit=1)
    
    def _check_sms_cooldown(self, phone):
        """检查短信发送冷却期
        
        Args:
            phone (str): 手机号
            
        Returns:
            dict or None: 如果在冷却期内返回错误响应，否则返回None
        """
        # 查找最近的验证码记录
        recent_sms = request.env['apollo.sms.verification.code'].sudo().search([
            ('phone', '=', phone),
            ('is_sent', '=', True)
        ], order='send_time desc', limit=1)
        
        if recent_sms and recent_sms.send_time:
            # 计算距离上次发送的时间
            time_diff = datetime.now() - recent_sms.send_time
            cooldown_seconds = 120  # 2分钟冷却期
            
            if time_diff.total_seconds() < cooldown_seconds:
                remaining_seconds = int(cooldown_seconds - time_diff.total_seconds())
                return self._error_response(
                    f'请等待{remaining_seconds}秒后再次发送验证码',
                    400
                )
        
        return None
    
    def _mask_phone_number(self, phone):
        """掩码手机号
        
        Args:
            phone (str): 手机号
            
        Returns:
            str: 掩码后的手机号
        """
        if len(phone) == 11:
            return f"{phone[:3]}****{phone[-4:]}"
        return phone
