# -*- coding: utf-8 -*-

import logging
from odoo import _
from odoo.http import Controller, request

_logger = logging.getLogger(__name__)


class BaseApiController(Controller):
    """API控制器基类，提供通用的响应和验证方法"""
    
    def _error_response(self, message, code=400, details=None):
        """通用错误响应方法
        
        Args:
            message (str): 错误消息
            code (int): HTTP状态码
            details (dict): 额外的错误详情
            
        Returns:
            dict: 标准化的错误响应
        """
        response = {
            'success': False,
            'message': message,
            'code': code
        }
        if details:
            response['details'] = details
        return response
    
    def _success_response(self, message, data=None, code=200):
        """通用成功响应方法
        
        Args:
            message (str): 成功消息
            data (dict): 响应数据
            code (int): HTTP状态码
            
        Returns:
            dict: 标准化的成功响应
        """
        response = {
            'success': True,
            'message': message,
            'code': code
        }
        if data is not None:
            response['data'] = data
        return response
    
    def _validate_json_data(self, required_fields=None, optional_fields=None):
        """验证JSON请求数据
        
        Args:
            required_fields (list): 必需字段列表
            optional_fields (list): 可选字段列表
            
        Returns:
            tuple: (data, error_response) 如果验证失败，error_response不为None
        """
        try:
            data = request.httprequest.get_json()
        except Exception as e:
            _logger.error(f"JSON解析失败: {e}")
            return None, self._error_response('无效的JSON格式')
        
        if not data:
            return None, self._error_response('请求数据不能为空')
        
        # 验证必需字段
        if required_fields:
            for field in required_fields:
                if field not in data or not data.get(field):
                    return None, self._error_response(f'字段 {field} 不能为空')
        
        # 过滤允许的字段
        if required_fields or optional_fields:
            allowed_fields = (required_fields or []) + (optional_fields or [])
            filtered_data = {k: v for k, v in data.items() if k in allowed_fields}
            return filtered_data, None
        
        return data, None
    
    def _validate_phone_number(self, phone):
        """验证手机号格式
        
        Args:
            phone (str): 手机号
            
        Returns:
            bool: 是否有效
        """
        if not phone:
            return False
        
        # 移除可能的空格和特殊字符
        phone = phone.strip().replace(' ', '').replace('-', '')
        
        # 检查是否为11位数字
        if not phone.isdigit() or len(phone) != 11:
            return False
        
        # 检查是否以1开头
        if not phone.startswith('1'):
            return False
        
        return True
    
    def _get_client_info(self):
        """获取客户端信息
        
        Returns:
            dict: 包含IP地址和User-Agent的字典
        """
        return {
            'ip_address': request.httprequest.environ.get('REMOTE_ADDR'),
            'user_agent': request.httprequest.environ.get('HTTP_USER_AGENT', '')
        }
    
    def _handle_exception(self, e, operation_name="操作"):
        """统一异常处理
        
        Args:
            e (Exception): 异常对象
            operation_name (str): 操作名称
            
        Returns:
            dict: 错误响应
        """
        error_msg = str(e)
        _logger.error(f"{operation_name}失败: {error_msg}")
        
        # 根据异常类型返回不同的错误消息
        if 'ValidationError' in str(type(e)):
            return self._error_response(error_msg, 400)
        elif 'AccessError' in str(type(e)):
            return self._error_response('权限不足', 403)
        else:
            return self._error_response(f'{operation_name}失败，请稍后重试', 500)
