
from odoo import api, fields, models, _
from odoo.exceptions import UserError

class ApolloOrganization(models.Model):
    _name = 'apollo.organization'
    _description = 'Apollo Organization'
    _rec_name = 'name'
    _order = 'name'
    
    # SQL 约束优化
    _sql_constraints = [
        ('unique_organization_name', 
         'UNIQUE(name)', 
         '组织名称必须唯一'),
    ]

    name = fields.Char(string='组织名称', required=True, translate=True)
    logo = fields.Binary(string="组织Logo")
    active = fields.Boolean(string='激活状态', default=True)
    
    # 组织所有者
    owner_id = fields.Many2one('res.users', string='组织所有者', readonly=True, index=True)
    
    # 所有者变更记录
    owner_change_ids = fields.One2many(
        'apollo.organization.owner.change', 
        'organization_id', 
        string='所有者变更记录'
    )
    owner_change_count = fields.Integer(
        string='变更记录数量', 
        compute='_compute_owner_change_count'
    )
    
    # 关联字段：该组织的所有邀请码
    invitation_code_ids = fields.One2many('apollo.invitation.code', 'organization_id', string='邀请码')
    invitation_code_count = fields.Integer(string='邀请码数量', compute='_compute_invitation_code_count')
    
    # 组织成员记录
    member_ids = fields.One2many('apollo.organization.member', 'organization_id', string='成员记录')
    active_member_ids = fields.One2many(
        'apollo.organization.member', 
        'organization_id',
        string='当前成员',
        domain=[('leave_date', '=', False)]
    )
    
    # 当前用户多对多字段（用于权限控制）
    current_user_ids = fields.Many2many(
        'res.users',
        'apollo_organization_user_rel',
        'organization_id',
        'user_id',
        string='当前用户',
        compute='_compute_current_user_ids',
        store=True,
        help='组织中的当前用户，用于权限控制'
    )
    
    @api.depends('active_member_ids.user_id', 'owner_id')
    def _compute_current_user_ids(self):
        """计算组织中的当前用户"""
        for record in self:
            current_users = record.active_member_ids.mapped('user_id')
            # 确保所有者也在当前用户列表中
            if record.owner_id and record.owner_id not in current_users:
                current_users |= record.owner_id
            record.current_user_ids = current_users
    
    @api.depends('owner_change_ids')
    def _compute_owner_change_count(self):
        """计算所有者变更记录数量"""
        for record in self:
            record.owner_change_count = len(record.owner_change_ids)
    
    @api.depends('invitation_code_ids')
    def _compute_invitation_code_count(self):
        """计算邀请码数量"""
        for record in self:
            record.invitation_code_count = len(record.invitation_code_ids)
    
    def generate_invitation_code(self):
        """为当前组织生成邀请码的向导"""
        return {
            'name': _('生成邀请码'),
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.invitation.code.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_organization_id': self.id,
            }
        }
    
    def create_invitation_code(self, phone):
        """直接创建邀请码"""
        return self.env['apollo.invitation.code'].create({
            'phone': phone,
            'organization_id': self.id,
        })
    
    @api.model_create_multi
    def create(self, vals_list):
        """创建组织时禁止设置owner_id，只能通过修改功能设置"""
        # 移除所有vals中的owner_id，防止在创建时设置
        for vals in vals_list:
            if 'owner_id' in vals:
                del vals['owner_id']
        
        return super(ApolloOrganization, self).create(vals_list)
    
    def change_owner_wizard(self):
        """打开设置/修改组织所有者的向导"""
        return {
            'name': _('设置/修改组织所有者'),
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.owner.change.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_organization_id': self.id,
                'default_current_owner_id': self.owner_id.id if self.owner_id else False,
            }
        }
    
    def change_owner(self, new_owner_id, reason=None):
        """修改组织所有者并记录变更"""
        self.ensure_one()
        
        if not new_owner_id:
            raise UserError(_('请选择新的所有者'))
        
        old_owner_id = self.owner_id.id if self.owner_id else False
        
        if old_owner_id == new_owner_id:
            raise UserError(_('新所有者与当前所有者相同'))
        
        # 直接更新owner_id（绕过readonly限制）
        self.env.cr.execute(
            "UPDATE apollo_organization SET owner_id = %s WHERE id = %s",
            (new_owner_id, self.id)
        )
        self.invalidate_recordset(['owner_id'])
        
        # 创建变更记录，根据是否有原owner确定原因
        default_reason = '设置组织初始所有者' if not old_owner_id else '修改组织所有者'
        
        self.env['apollo.organization.owner.change'].create({
            'organization_id': self.id,
            'old_owner_id': old_owner_id,
            'new_owner_id': new_owner_id,
            'changed_by_id': self.env.user.id,
            'reason': reason or default_reason
        })
        
        return True
    
    def view_owner_change_history(self):
        """查看所有者变更历史"""
        return {
            'name': _('所有者变更历史'),
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.owner.change',
            'view_mode': 'list,form',
            'domain': [('organization_id', '=', self.id)],
            'context': {'default_organization_id': self.id}
        }
    
    def add_member_wizard(self):
        """添加成员向导"""
        return {
            'name': _('添加组织成员'),
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.member.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {
                'default_organization_id': self.id,
                'default_action': 'add',
            }
        }
    
    def view_member_history(self):
        """查看成员历史"""
        return {
            'name': _('成员历史记录'),
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.member',
            'view_mode': 'list,form',
            'domain': [('organization_id', '=', self.id)],
            'context': {'default_organization_id': self.id}
        }
    
    def get_active_members(self):
        """获取当前活跃成员"""
        return self.env['apollo.organization.member'].get_active_members(self.id)
    
    def is_user_owner(self, user=None):
        """检查用户是否为组织所有者"""
        self.ensure_one()
        if not user:
            user = self.env.user
        return self.owner_id and self.owner_id.id == user.id
    
    def is_user_member(self, user=None):
        """检查用户是否为组织成员"""
        self.ensure_one()
        if not user:
            user = self.env.user
        return user in self.current_user_ids
    
    def is_user_business_admin(self, user=None):
        """检查用户是否为业务管理员"""
        if not user:
            user = self.env.user
        business_admin_group = self.env.ref('apollo_account.group_apollo_business_admin', raise_if_not_found=False)
        return business_admin_group and business_admin_group in user.groups_id
    
    def check_user_access(self, access_type='read', user=None):
        """检查用户对组织的访问权限
        
        Args:
            access_type: 'read', 'write', 'create', 'unlink'
            user: 用户记录，默认为当前用户
        
        Returns:
            bool: 是否有权限
        """
        self.ensure_one()
        if not user:
            user = self.env.user
        
        # 超级用户有所有权限
        if user._is_superuser():
            return True
        
        # 业务管理员有所有权限
        if self.is_user_business_admin(user):
            return True
        
        # 组织所有者有所有权限
        if self.is_user_owner(user):
            return True
        
        # 组织成员只有读权限
        if access_type == 'read' and self.is_user_member(user):
            return True
        
        return False
    
    def assign_user_groups(self):
        """为组织用户分配相应的安全组"""
        self.ensure_one()
        
        # 获取安全组
        owner_group = self.env.ref('apollo_account.group_apollo_organization_owner', raise_if_not_found=False)
        member_group = self.env.ref('apollo_account.group_apollo_organization_member', raise_if_not_found=False)
        business_admin_group = self.env.ref('apollo_account.group_apollo_business_admin', raise_if_not_found=False)
        
        if not owner_group or not member_group:
            return False
        
        # 为所有者分配所有者组（如果不是业务管理员）
        if self.owner_id:
            # 检查是否已经是业务管理员
            if not self.is_user_business_admin(self.owner_id):
                if owner_group not in self.owner_id.groups_id:
                    self.owner_id.groups_id = [(4, owner_group.id)]
        
        # 为成员分配成员组（如果不是业务管理员或所有者）
        for member in self.active_member_ids:
            # 跳过业务管理员和所有者
            if self.is_user_business_admin(member.user_id) or member.user_id == self.owner_id:
                continue
            
            if member_group not in member.user_id.groups_id:
                member.user_id.groups_id = [(4, member_group.id)]
        
        return True
    
    def remove_user_groups(self, user):
        """移除用户的组织相关安全组"""
        owner_group = self.env.ref('apollo_account.group_apollo_organization_owner', raise_if_not_found=False)
        member_group = self.env.ref('apollo_account.group_apollo_organization_member', raise_if_not_found=False)
        business_admin_group = self.env.ref('apollo_account.group_apollo_business_admin', raise_if_not_found=False)
        
        if not owner_group or not member_group:
            return False
        
        # 业务管理员不移除任何组
        if self.is_user_business_admin(user):
            return True
        
        # 检查用户是否还在其他组织中
        other_orgs = self.env['apollo.organization'].search([
            ('id', '!=', self.id),
            ('current_user_ids', 'in', [user.id])
        ])
        
        # 如果用户不在其他组织中，移除相关组
        if not other_orgs:
            if owner_group in user.groups_id:
                user.groups_id = [(3, owner_group.id)]
            if member_group in user.groups_id:
                user.groups_id = [(3, member_group.id)]
        
        return True
    
    def write(self, vals):
        """重写write方法，在所有者变更时自动管理用户组"""
        result = super(ApolloOrganization, self).write(vals)
        
        # 如果所有者发生变更，重新分配用户组
        if 'owner_id' in vals:
            for record in self:
                record.assign_user_groups()
        
        return result
