# -*- coding: utf-8 -*-

import logging
from odoo import api, fields, models, _
from odoo.exceptions import ValidationError
from ..utils.validators import PhoneValidator, InvitationCodeValidator
from ..utils.helpers import CodeGenerator, DateTimeHelper

_logger = logging.getLogger(__name__)


class InvitationCode(models.Model):
    """邀请码模型 - 用于组织邀请和用户注册"""
    
    _name = 'apollo.invitation.code'
    _description = 'Apollo Invitation Code'
    _rec_name = 'code'
    _order = 'create_date desc'
    _sql_constraints = [
        ('unique_code', 'UNIQUE(code)', '邀请码必须唯一'),
    ]

    # 基础字段
    code = fields.Char(
        string='邀请码',
        required=True,
        readonly=True,
        default=lambda self: self._generate_unique_code(),
        help='12位唯一邀请码，由系统自动生成'
    )
    
    phone = fields.Char(
        string='手机号',
        required=True,
        index=True,
        help='绑定的手机号，用于接收验证码'
    )
    
    organization_id = fields.Many2one(
        'apollo.organization',
        string='组织',
        required=True,
        ondelete='cascade',
        index=True,
        help='关联的组织'
    )
    
    # 关联字段
    organization_name = fields.Char(
        related='organization_id.name',
        string='组织名称',
        readonly=True
    )
    
    # 状态字段
    active = fields.Boolean(
        string='激活状态',
        default=True,
        help='是否激活，停用后无法使用'
    )
    
    voided = fields.Boolean(
        string='已作废',
        default=False,
        help='是否已作废，作废后无法使用'
    )
    
    void_date = fields.Datetime(
        string='作废时间',
        readonly=True,
        help='邀请码作废的时间'
    )
    
    # 用户绑定
    bound_user_id = fields.Many2one(
        'res.users',
        string='绑定用户',
        readonly=True,
        help='使用此邀请码注册的用户'
    )
    
    bound_user_name = fields.Char(
        related='bound_user_id.name',
        string='绑定用户名',
        readonly=True
    )
    
    bound_date = fields.Datetime(
        string='绑定时间',
        readonly=True,
        help='用户绑定的时间'
    )
    
    # 创建信息
    created_by = fields.Many2one(
        'res.users',
        string='创建人',
        default=lambda self: self.env.user,
        readonly=True
    )
    
    # 计算字段
    is_used = fields.Boolean(
        string='已使用',
        compute='_compute_is_used',
        store=True,
        help='是否已被用户使用'
    )
    
    status = fields.Selection([
        ('active', '有效'),
        ('used', '已使用'),
        ('voided', '已作废'),
        ('inactive', '已停用')
    ], string='状态', compute='_compute_status', store=True)
    
    # 统计字段
    sms_count = fields.Integer(
        string='短信发送次数',
        compute='_compute_sms_count',
        help='此邀请码相关的短信发送次数'
    )

    @api.depends('bound_user_id')
    def _compute_is_used(self):
        """计算是否已使用"""
        for record in self:
            record.is_used = bool(record.bound_user_id)

    @api.depends('active', 'voided', 'is_used')
    def _compute_status(self):
        """计算状态"""
        for record in self:
            if record.voided:
                record.status = 'voided'
            elif not record.active:
                record.status = 'inactive'
            elif record.is_used:
                record.status = 'used'
            else:
                record.status = 'active'

    def _compute_sms_count(self):
        """计算短信发送次数"""
        for record in self:
            record.sms_count = self.env['apollo.sms.verification.code'].search_count([
                ('invitation_code', '=', record.code)
            ])

    @api.model
    def _generate_unique_code(self):
        """生成唯一的邀请码
        
        Returns:
            str: 唯一的12位邀请码
        """
        max_attempts = 10
        for attempt in range(max_attempts):
            code = CodeGenerator.generate_invitation_code()
            
            # 检查唯一性
            if not self.search([('code', '=', code)], limit=1):
                return code
        
        # 如果多次尝试都失败，使用时间戳确保唯一性
        import time
        timestamp = str(int(time.time()))[-6:]
        base_code = CodeGenerator.generate_invitation_code(6)
        return f"{base_code}{timestamp}"

    @api.constrains('phone')
    def _check_phone_format(self):
        """验证手机号格式"""
        for record in self:
            if not PhoneValidator.is_valid_china_mobile(record.phone):
                raise ValidationError(_('手机号格式不正确，请输入有效的11位手机号'))

    @api.constrains('phone', 'organization_id', 'active', 'voided')
    def _check_duplicate_active_invitation(self):
        """检查重复的有效邀请码"""
        for record in self:
            if record.active and not record.voided:
                existing = self.search([
                    ('phone', '=', record.phone),
                    ('organization_id', '=', record.organization_id.id),
                    ('active', '=', True),
                    ('voided', '=', False),
                    ('id', '!=', record.id)
                ], limit=1)
                
                if existing:
                    raise ValidationError(
                        _('该手机号在此组织中已有有效的邀请码: %s') % existing.code
                    )

    @api.model_create_multi
    def create(self, vals_list):
        """创建邀请码记录"""
        # 清理和验证手机号
        for vals in vals_list:
            if 'phone' in vals:
                vals['phone'] = PhoneValidator.clean_phone_number(vals['phone'])
        
        records = super().create(vals_list)
        
        # 记录创建日志
        for record in records:
            _logger.info(f"创建邀请码: {record.code}, 手机号: {record.phone}, 组织: {record.organization_name}")
        
        return records

    def write(self, vals):
        """更新邀请码记录"""
        # 清理手机号
        if 'phone' in vals:
            vals['phone'] = PhoneValidator.clean_phone_number(vals['phone'])
        
        result = super().write(vals)
        
        # 记录更新日志
        if 'voided' in vals and vals['voided']:
            for record in self:
                _logger.info(f"作废邀请码: {record.code}")
        
        return result

    def void_code(self):
        """作废邀请码
        
        Returns:
            bool: 操作是否成功
        """
        self.ensure_one()
        
        if self.voided:
            raise ValidationError(_('邀请码已经作废'))
        
        if self.is_used:
            raise ValidationError(_('邀请码已被使用，无法作废'))
        
        self.write({
            'voided': True,
            'void_date': DateTimeHelper.now()
        })
        
        _logger.info(f"邀请码已作废: {self.code}")
        return True

    def bind_user(self, user_id):
        """绑定用户到邀请码
        
        Args:
            user_id (int): 用户ID
            
        Returns:
            bool: 绑定是否成功
        """
        self.ensure_one()
        
        if self.voided:
            raise ValidationError(_('邀请码已作废，无法绑定用户'))
        
        if not self.active:
            raise ValidationError(_('邀请码已停用，无法绑定用户'))
        
        if self.bound_user_id:
            raise ValidationError(_('邀请码已绑定用户: %s') % self.bound_user_id.name)
        
        # 验证用户存在
        user = self.env['res.users'].browse(user_id)
        if not user.exists():
            raise ValidationError(_('用户不存在'))
        
        # 验证手机号匹配
        if user.mobile and user.mobile != self.phone:
            _logger.warning(f"用户手机号({user.mobile})与邀请码手机号({self.phone})不匹配")
        
        # 绑定用户
        self.write({
            'bound_user_id': user_id,
            'bound_date': DateTimeHelper.now()
        })
        
        # 创建组织成员记录
        self._create_organization_member(user_id)
        
        # 如果组织没有所有者，设置为所有者
        if not self.organization_id.owner_id:
            self.organization_id.write({'owner_id': user_id})
            _logger.info(f"设置用户 {user.name} 为组织 {self.organization_id.name} 的所有者")
        
        _logger.info(f"邀请码绑定成功: {self.code} -> 用户: {user.name}")
        return True

    def _create_organization_member(self, user_id):
        """创建组织成员记录
        
        Args:
            user_id (int): 用户ID
        """
        # 检查是否已存在成员记录
        existing_member = self.env['apollo.organization.member'].search([
            ('user_id', '=', user_id),
            ('organization_id', '=', self.organization_id.id),
            ('leave_date', '=', False)
        ], limit=1)
        
        if not existing_member:
            self.env['apollo.organization.member'].create({
                'user_id': user_id,
                'organization_id': self.organization_id.id,
                'join_date': DateTimeHelper.now(),
                'invitation_code_id': self.id
            })

    def get_usage_info(self):
        """获取邀请码使用信息
        
        Returns:
            dict: 使用信息
        """
        self.ensure_one()
        
        return {
            'code': self.code,
            'phone': self.phone,
            'masked_phone': PhoneValidator.mask_phone_number(self.phone),
            'organization': {
                'id': self.organization_id.id,
                'name': self.organization_id.name
            },
            'status': self.status,
            'is_used': self.is_used,
            'bound_user': {
                'id': self.bound_user_id.id,
                'name': self.bound_user_id.name
            } if self.bound_user_id else None,
            'create_date': self.create_date.isoformat() if self.create_date else None,
            'bound_date': self.bound_date.isoformat() if self.bound_date else None,
            'void_date': self.void_date.isoformat() if self.void_date else None,
            'sms_count': self.sms_count
        }

    @api.model
    def cleanup_expired_codes(self, days=30):
        """清理过期的邀请码
        
        Args:
            days (int): 过期天数
            
        Returns:
            int: 清理的记录数
        """
        cutoff_date = DateTimeHelper.add_days(DateTimeHelper.now(), -days)
        
        expired_codes = self.search([
            ('create_date', '<', cutoff_date),
            ('bound_user_id', '=', False),
            ('active', '=', False)
        ])
        
        count = len(expired_codes)
        if count > 0:
            expired_codes.unlink()
            _logger.info(f"清理了 {count} 个过期邀请码")
        
        return count

    def action_view_sms_records(self):
        """查看相关的短信记录"""
        self.ensure_one()
        
        return {
            'type': 'ir.actions.act_window',
            'name': f'邀请码 {self.code} 的短信记录',
            'res_model': 'apollo.sms.verification.code',
            'view_mode': 'list,form',
            'domain': [('invitation_code', '=', self.code)],
            'context': {'default_invitation_code': self.code}
        }