import random
import logging
from datetime import datetime, timedelta
from odoo import api, fields, models, _
from odoo.exceptions import ValidationError

_logger = logging.getLogger(__name__)


class SmsVerificationCode(models.Model):
    _name = 'apollo.sms.verification.code'
    _description = 'SMS Verification Code'
    _rec_name = 'phone'
    _order = 'create_date desc'

    phone = fields.Char(string='手机号', required=True, index=True)
    code = fields.Char(string='验证码', required=True, size=6)
    invitation_code = fields.Char(string='邀请码', index=True)
    organization_id = fields.Many2one('apollo.organization', string='组织', ondelete='cascade')
    verification_type = fields.Selection([
        ('invitation', '邀请码验证'),
        ('login', '登录验证')
    ], string='验证类型', required=True, default='invitation')
    organization_name = fields.Char(related='organization_id.name', string='组织名称', readonly=True)
    
    # 发送状态和时间
    is_sent = fields.Boolean(string='是否发送成功', default=False)
    send_time = fields.Datetime(string='发送时间')
    error_message = fields.Text(string='错误信息')
    
    # 有效期和状态
    expires_at = fields.Datetime(string='过期时间', required=True)
    is_expired = fields.Boolean(string='是否过期', compute='_compute_is_expired', store=True)
    is_used = fields.Boolean(string='是否已使用', default=False)
    used_at = fields.Datetime(string='使用时间')
    
    # 创建信息
    created_by = fields.Many2one('res.users', string='创建人', default=lambda self: self.env.user)
    ip_address = fields.Char(string='IP地址')
    user_agent = fields.Text(string='用户代理')
    
    @api.depends('expires_at')
    def _compute_is_expired(self):
        """计算是否过期"""
        now = fields.Datetime.now()
        for record in self:
            record.is_expired = record.expires_at < now if record.expires_at else True
    
    @api.model
    def _generate_code(self):
        """生成6位随机验证码"""
        return str(random.randint(100000, 999999))
    
    @api.constrains('phone')
    def _check_phone(self):
        """验证手机号格式"""
        for record in self:
            if record.phone and not record.phone.isdigit():
                raise ValidationError(_('手机号只能包含数字'))
            if record.phone and len(record.phone) != 11:
                raise ValidationError(_('手机号必须为11位数字'))
    
    @api.constrains('code')
    def _check_code(self):
        """验证验证码格式"""
        for record in self:
            if record.code and (not record.code.isdigit() or len(record.code) != 6):
                raise ValidationError(_('验证码必须为6位数字'))
    
    @api.model
    def create_verification_code(self, phone, invitation_code, organization_id, ip_address=None, user_agent=None):
        """创建验证码（用于邀请码注册）"""
        # 检查5分钟内是否已经发送过验证码
        five_minutes_ago = datetime.now() - timedelta(minutes=5)
        existing_code = self.search([
            ('phone', '=', phone),
            ('invitation_code', '=', invitation_code),
            ('verification_type', '=', 'invitation'),
            ('create_date', '>', five_minutes_ago),
            ('is_sent', '=', True)
        ], limit=1)
        
        if existing_code:
            remaining_time = 300 - (datetime.now() - existing_code.create_date).seconds
            raise ValidationError(_('请等待%d秒后再次发送验证码') % remaining_time)
        
        # 生成验证码
        code = self._generate_code()
        expires_at = datetime.now() + timedelta(minutes=5)
        
        # 创建验证码记录
        verification_record = self.create({
            'phone': phone,
            'code': code,
            'invitation_code': invitation_code,
            'organization_id': organization_id,
            'expires_at': expires_at,
            'ip_address': ip_address,
            'user_agent': user_agent,
            'verification_type': 'invitation',
        })
        
        return verification_record
    
    @api.model
    def create_login_verification_code(self, phone, ip_address=None, user_agent=None):
        """创建登录验证码"""
        # 检查5分钟内是否已经发送过登录验证码
        five_minutes_ago = datetime.now() - timedelta(minutes=5)
        existing_code = self.search([
            ('phone', '=', phone),
            ('verification_type', '=', 'login'),
            ('create_date', '>', five_minutes_ago),
            ('is_sent', '=', True)
        ], limit=1)
        
        if existing_code:
            remaining_time = 300 - (datetime.now() - existing_code.create_date).seconds
            raise ValidationError(_('请等待%d秒后再次发送验证码') % remaining_time)
        
        # 生成验证码
        code = self._generate_code()
        expires_at = datetime.now() + timedelta(minutes=5)
        
        # 创建验证码记录
        verification_record = self.create({
            'phone': phone,
            'code': code,
            'expires_at': expires_at,
            'ip_address': ip_address,
            'user_agent': user_agent,
            'verification_type': 'login',
        })
        
        return verification_record
    
    def send_sms(self):
        """发送短信验证码"""
        try:
            # 调用短信发送服务
            sms_service = self.env['apollo.sms.service']
            success, error_msg = sms_service.send_verification_code(self.phone, self.code)
            
            # 更新发送状态
            self.write({
                'is_sent': success,
                'send_time': fields.Datetime.now(),
                'error_message': error_msg if not success else None
            })
            
            _logger.info(f"验证码发送{'成功' if success else '失败'}: {self.phone}, 验证码: {self.code}")
            
            return success, error_msg
            
        except Exception as e:
            error_msg = str(e)
            self.write({
                'is_sent': False,
                'send_time': fields.Datetime.now(),
                'error_message': error_msg
            })
            _logger.error(f"发送验证码时发生异常: {error_msg}")
            return False, error_msg
    
    def mark_as_used(self):
        """标记验证码为已使用"""
        if self.is_used:
            raise ValidationError(_('验证码已经使用过'))
        
        if self.is_expired:
            raise ValidationError(_('验证码已过期'))
        
        self.write({
            'is_used': True,
            'used_at': fields.Datetime.now()
        })
    
    @api.model
    def verify_code(self, phone, code, invitation_code):
        """验证验证码（邀请码注册）"""
        verification = self.search([
            ('phone', '=', phone),
            ('code', '=', code),
            ('invitation_code', '=', invitation_code),
            ('verification_type', '=', 'invitation'),
            ('is_used', '=', False),
            ('is_sent', '=', True)
        ], limit=1)
        
        if not verification:
            return False, _('验证码不正确或已使用')
        
        if verification.is_expired:
            return False, _('验证码已过期')
        
        # 标记为已使用
        verification.mark_as_used()
        
        return True, _('验证成功')
    
    @api.model
    def verify_login_code(self, phone, code):
        """验证登录验证码"""
        verification = self.search([
            ('phone', '=', phone),
            ('code', '=', code),
            ('verification_type', '=', 'login'),
            ('is_used', '=', False),
            ('is_sent', '=', True)
        ], limit=1)
        
        if not verification:
            return False, _('验证码不正确或已使用')
        
        if verification.is_expired:
            return False, _('验证码已过期')
        
        # 标记为已使用
        verification.mark_as_used()
        
        return True, _('验证成功')
    
    @api.model
    def cleanup_expired_codes(self):
        """清理过期的验证码（可用于定时任务）"""
        expired_codes = self.search([
            ('is_expired', '=', True),
            ('create_date', '<', fields.Datetime.now() - timedelta(days=7))
        ])
        if expired_codes:
            _logger.info(f"清理了 {len(expired_codes)} 条过期验证码记录")
            expired_codes.unlink()
