# -*- coding: utf-8 -*-

import re
import logging

_logger = logging.getLogger(__name__)


class PhoneValidator:
    """手机号验证器"""
    
    # 中国大陆手机号正则表达式
    CHINA_MOBILE_PATTERN = re.compile(r'^1[3-9]\d{9}$')
    
    @classmethod
    def is_valid_china_mobile(cls, phone):
        """验证中国大陆手机号
        
        Args:
            phone (str): 手机号
            
        Returns:
            bool: 是否有效
        """
        if not phone:
            return False
        
        # 清理手机号（移除空格、横线等）
        cleaned_phone = cls.clean_phone_number(phone)
        
        # 验证格式
        return bool(cls.CHINA_MOBILE_PATTERN.match(cleaned_phone))
    
    @classmethod
    def clean_phone_number(cls, phone):
        """清理手机号格式
        
        Args:
            phone (str): 原始手机号
            
        Returns:
            str: 清理后的手机号
        """
        if not phone:
            return ''
        
        # 移除常见的分隔符和空格
        cleaned = re.sub(r'[\s\-\(\)\+]', '', str(phone))
        
        # 如果以+86开头，移除国家代码
        if cleaned.startswith('+86'):
            cleaned = cleaned[3:]
        elif cleaned.startswith('86') and len(cleaned) == 13:
            cleaned = cleaned[2:]
        
        return cleaned
    
    @classmethod
    def mask_phone_number(cls, phone):
        """掩码手机号
        
        Args:
            phone (str): 手机号
            
        Returns:
            str: 掩码后的手机号
        """
        cleaned_phone = cls.clean_phone_number(phone)
        
        if len(cleaned_phone) == 11:
            return f"{cleaned_phone[:3]}****{cleaned_phone[-4:]}"
        elif len(cleaned_phone) >= 7:
            # 对于其他长度的号码，保留前3位和后4位
            return f"{cleaned_phone[:3]}****{cleaned_phone[-4:]}"
        else:
            # 太短的号码不进行掩码
            return phone


class InvitationCodeValidator:
    """邀请码验证器"""
    
    # 邀请码格式：12位大写字母和数字组合
    CODE_PATTERN = re.compile(r'^[A-Z0-9]{12}$')
    
    @classmethod
    def is_valid_format(cls, code):
        """验证邀请码格式
        
        Args:
            code (str): 邀请码
            
        Returns:
            bool: 是否有效
        """
        if not code:
            return False
        
        return bool(cls.CODE_PATTERN.match(code.upper()))
    
    @classmethod
    def normalize_code(cls, code):
        """标准化邀请码格式
        
        Args:
            code (str): 原始邀请码
            
        Returns:
            str: 标准化后的邀请码
        """
        if not code:
            return ''
        
        # 转换为大写并移除空格
        return code.upper().strip()


class SmsCodeValidator:
    """短信验证码验证器"""
    
    # 验证码格式：6位数字
    CODE_PATTERN = re.compile(r'^\d{6}$')
    
    @classmethod
    def is_valid_format(cls, code):
        """验证短信验证码格式
        
        Args:
            code (str): 验证码
            
        Returns:
            bool: 是否有效
        """
        if not code:
            return False
        
        return bool(cls.CODE_PATTERN.match(str(code)))
    
    @classmethod
    def normalize_code(cls, code):
        """标准化验证码格式
        
        Args:
            code (str): 原始验证码
            
        Returns:
            str: 标准化后的验证码
        """
        if not code:
            return ''
        
        # 移除空格并确保是字符串
        return str(code).strip()


class UserDataValidator:
    """用户数据验证器"""
    
    # 用户名格式：2-50个字符，支持中文、英文、数字
    NAME_PATTERN = re.compile(r'^[\u4e00-\u9fa5a-zA-Z0-9\s]{2,50}$')
    
    # 邮箱格式
    EMAIL_PATTERN = re.compile(r'^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$')
    
    @classmethod
    def is_valid_name(cls, name):
        """验证用户名格式
        
        Args:
            name (str): 用户名
            
        Returns:
            bool: 是否有效
        """
        if not name:
            return False
        
        # 去除首尾空格后验证
        cleaned_name = name.strip()
        return bool(cls.NAME_PATTERN.match(cleaned_name))
    
    @classmethod
    def is_valid_email(cls, email):
        """验证邮箱格式
        
        Args:
            email (str): 邮箱地址
            
        Returns:
            bool: 是否有效
        """
        if not email:
            return True  # 邮箱可以为空
        
        return bool(cls.EMAIL_PATTERN.match(email.strip().lower()))
    
    @classmethod
    def clean_name(cls, name):
        """清理用户名
        
        Args:
            name (str): 原始用户名
            
        Returns:
            str: 清理后的用户名
        """
        if not name:
            return ''
        
        # 去除首尾空格，压缩中间的多个空格为单个空格
        return ' '.join(name.strip().split())
    
    @classmethod
    def clean_email(cls, email):
        """清理邮箱地址
        
        Args:
            email (str): 原始邮箱
            
        Returns:
            str: 清理后的邮箱
        """
        if not email:
            return ''
        
        return email.strip().lower()


def validate_request_data(data, required_fields=None, optional_fields=None, validators=None):
    """通用请求数据验证函数
    
    Args:
        data (dict): 请求数据
        required_fields (list): 必需字段列表
        optional_fields (list): 可选字段列表
        validators (dict): 字段验证器映射 {field_name: validator_function}
        
    Returns:
        tuple: (is_valid: bool, errors: list, cleaned_data: dict)
    """
    errors = []
    cleaned_data = {}
    
    if not isinstance(data, dict):
        return False, ['请求数据格式错误'], {}
    
    # 验证必需字段
    if required_fields:
        for field in required_fields:
            if field not in data or data[field] is None or data[field] == '':
                errors.append(f'字段 {field} 不能为空')
            else:
                cleaned_data[field] = data[field]
    
    # 处理可选字段
    if optional_fields:
        for field in optional_fields:
            if field in data and data[field] is not None:
                cleaned_data[field] = data[field]
    
    # 应用自定义验证器
    if validators:
        for field_name, validator_func in validators.items():
            if field_name in cleaned_data:
                try:
                    is_valid = validator_func(cleaned_data[field_name])
                    if not is_valid:
                        errors.append(f'字段 {field_name} 格式不正确')
                except Exception as e:
                    _logger.error(f"验证字段 {field_name} 时出错: {str(e)}")
                    errors.append(f'字段 {field_name} 验证失败')
    
    return len(errors) == 0, errors, cleaned_data
