from odoo import api, fields, models, _
from odoo.exceptions import UserError


class ApolloOrganizationOwnerChangeWizard(models.TransientModel):
    _name = 'apollo.organization.owner.change.wizard'
    _description = 'Apollo Organization Owner Change Wizard'

    organization_id = fields.Many2one(
        'apollo.organization',
        string='组织',
        required=True,
        readonly=True
    )
    current_owner_id = fields.Many2one(
        'res.users',
        string='当前所有者',
        readonly=True,
        help='如果为空，表示组织当前没有所有者'
    )
    new_owner_id = fields.Many2one(
        'res.users',
        string='新所有者',
        required=True,
        domain=[('active', '=', True)]
    )
    reason = fields.Text(
        string='变更原因',
        required=True,
        help='请说明修改组织所有者的原因'
    )
    
    @api.model
    def default_get(self, fields_list):
        """设置默认值"""
        res = super().default_get(fields_list)
        if 'current_owner_id' in fields_list:
            organization_id = self.env.context.get('default_organization_id')
            if organization_id:
                organization = self.env['apollo.organization'].browse(organization_id)
                res['current_owner_id'] = organization.owner_id.id if organization.owner_id else False
        return res
    
    @api.onchange('new_owner_id')
    def _onchange_new_owner_id(self):
        """检查新所有者是否与当前所有者相同"""
        if self.new_owner_id and self.current_owner_id and self.new_owner_id.id == self.current_owner_id.id:
            return {
                'warning': {
                    'title': _('警告'),
                    'message': _('新所有者与当前所有者相同，请选择不同的用户。')
                }
            }
    
    def confirm_change_owner(self):
        """确认修改组织所有者"""
        self.ensure_one()
        
        if not self.new_owner_id:
            raise UserError(_('请选择新的所有者'))
        
        if not self.reason.strip():
            raise UserError(_('请填写变更原因'))
        
        if self.current_owner_id and self.new_owner_id.id == self.current_owner_id.id:
            raise UserError(_('新所有者与当前所有者相同'))
        
        # 执行所有者变更
        self.organization_id.change_owner(self.new_owner_id.id, self.reason)
        
        # 根据是否有当前所有者，显示不同的成功消息
        if self.current_owner_id:
            message = _('组织所有者已成功修改为：%s') % self.new_owner_id.name
        else:
            message = _('组织所有者已成功设置为：%s') % self.new_owner_id.name
        
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': _('成功'),
                'message': message,
                'type': 'success',
                'sticky': False,
            }
        }
