/**
 * Apollo API 工具模块
 * 
 * 提供与 Apollo Account API 的现代化交互功能
 */

import { _t } from "@web/core/l10n/translation";

// API 基础配置
const API_BASE = '/api/apollo';
const STORAGE_KEY = 'apollo_token';
const STORAGE_USER_KEY = 'apollo_user_info';

/**
 * API 工具类 - 现代化版本
 */
export class ApolloAPI {

    /**
     * 发起API请求
     * @param {string} endpoint API端点
     * @param {Object} data 请求数据
     * @param {boolean} showLoading 是否显示加载状态
     * @returns {Promise<Object>} API响应
     */
    static async request(endpoint, data = {}, showLoading = true) {
        console.log('Apollo API: Request to', endpoint, 'with data:', data);

        try {
            const response = await fetch(`${API_BASE}${endpoint}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(data)
            });

            console.log('Apollo API: Response status:', response.status);
            const result = await response.json();
            console.log('Apollo API: Response data:', result);

            // 处理JSON-RPC格式响应
            const responseData = result.result || result;

            if (!responseData.success) {
                const errorMessage = responseData.message || _t('Request failed');

                // 创建错误对象，包含更多错误信息
                const apiError = new Error(errorMessage);
                apiError.code = responseData.code || 'UNKNOWN_ERROR';
                apiError.details = responseData.details || null;

                throw apiError;
            }

            return responseData;
        } catch (error) {
            console.error('Apollo API: Request error:', error);
            throw error;
        }
    }

    /**
     * 用户登录 - 密码方式
     * @param {string} login 手机号或邮箱
     * @param {string} password 密码
     * @returns {Promise<Object>}
     */
    static async loginWithPassword(login, password) {
        const data = { password };

        // 判断是手机号还是邮箱
        if (login.includes('@')) {
            data.email = login;
        } else {
            data.phone = login;
        }

        return this.request('/user/login', data);
    }

    /**
     * 用户登录 - 短信验证码方式
     * @param {string} phone 手机号
     * @param {string} smsCode 短信验证码
     * @returns {Promise<Object>}
     */
    static async loginWithSms(phone, smsCode) {
        return this.request('/user/login/sms', {
            phone: phone,
            sms_code: smsCode
        });
    }

    /**
     * 发送登录短信验证码
     * @param {string} phone 手机号
     * @returns {Promise<Object>}
     */
    static async sendLoginSms(phone) {
        return this.request('/sms/send_login_code', { phone });
    }

    /**
     * 获取邀请码信息
     * @param {string} invitationCode 邀请码
     * @returns {Promise<Object>}
     */
    static async getInvitationInfo(invitationCode) {
        return this.request('/invitation/info', {
            invitation_code: invitationCode
        });
    }

    /**
     * 发送邀请注册短信验证码
     * @param {string} invitationCode 邀请码
     * @returns {Promise<Object>}
     */
    static async sendInvitationSms(invitationCode) {
        return this.request('/sms/send_invitation_code', {
            invitation_code: invitationCode
        });
    }

    /**
     * 验证邀请注册短信验证码
     * @param {string} invitationCode 邀请码
     * @param {string} smsCode 短信验证码
     * @returns {Promise<Object>}
     */
    static async verifyInvitationSms(invitationCode, smsCode) {
        return this.request('/sms/verify_invitation_code', {
            invitation_code: invitationCode,
            sms_code: smsCode
        });
    }

    /**
     * 完成用户注册
     * @param {string} token 注册令牌
     * @param {string} name 用户姓名
     * @param {string} password 密码
     * @returns {Promise<Object>}
     */
    static async register(token, name, password) {
        return this.request('/user/register', {
            token: token,
            name: name,
            password: password
        });
    }

    /**
     * 获取用户档案信息（个人基本信息）
     * @param {string} token JWT令牌
     * @returns {Promise<Object>}
     */
    static async getUserProfile(token) {
        return this.request('/user/profile', { token }, false);
    }

    /**
     * 获取用户组织信息（组织成员信息）
     * @param {string} token JWT令牌
     * @returns {Promise<Object>}
     */
    static async getUserOrganizations(token) {
        return this.request('/user/organizations', { token }, false);
    }
}

/**
 * 存储工具类
 */
export class ApolloStorage {

    /**
     * 保存JWT令牌
     * @param {string} token 
     */
    static saveToken(token) {
        localStorage.setItem(STORAGE_KEY, token);
    }

    /**
     * 获取JWT令牌
     * @returns {string|null}
     */
    static getToken() {
        return localStorage.getItem(STORAGE_KEY);
    }

    /**
     * 清除所有存储数据
     */
    static clear() {
        localStorage.removeItem(STORAGE_KEY);
        localStorage.removeItem(STORAGE_USER_KEY);
    }

    /**
     * 保存用户信息
     * @param {Object} userInfo 
     */
    static saveUserInfo(userInfo) {
        localStorage.setItem(STORAGE_USER_KEY, JSON.stringify(userInfo));
    }

    /**
     * 获取用户信息
     * @returns {Object|null}
     */
    static getUserInfo() {
        const userInfo = localStorage.getItem(STORAGE_USER_KEY);
        return userInfo ? JSON.parse(userInfo) : null;
    }
}

/**
 * 工具函数
 */
export class ApolloUtils {

    /**
     * 格式化手机号显示
     * @param {string} phone 
     * @returns {string}
     */
    static formatPhone(phone) {
        if (!phone) return '';
        const cleaned = phone.replace(/\D/g, '');
        if (cleaned.length === 11 && cleaned.startsWith('1')) {
            return cleaned.replace(/(\d{3})(\d{4})(\d{4})/, '$1****$3');
        }
        return phone;
    }

    /**
     * 格式化日期显示
     * @param {string} dateString 
     * @returns {string}
     */
    static formatDate(dateString) {
        if (!dateString) return '';
        const date = new Date(dateString);
        return date.toLocaleDateString('zh-CN');
    }

    /**
     * 获取姓名首字母
     * @param {string} name 
     * @returns {string}
     */
    static getInitials(name) {
        if (!name) return '?';
        const words = name.trim().split(' ');
        if (words.length >= 2) {
            return (words[0][0] + words[1][0]).toUpperCase();
        }
        return name[0].toUpperCase();
    }

    /**
     * 解析JWT令牌过期时间
     * @param {string} token 
     * @returns {Date|null}
     */
    static parseTokenExpiry(token) {
        try {
            const payload = JSON.parse(atob(token.split('.')[1]));
            return payload.exp ? new Date(payload.exp * 1000) : null;
        } catch (e) {
            console.warn('Apollo Utils: Cannot parse token expiry time');
            return null;
        }
    }

    /**
     * 倒计时功能
     * @param {HTMLButtonElement} button 按钮元素
     * @param {number} seconds 秒数
     * @param {string} originalText 原始文本
     */
    static startCountdown(button, seconds, originalText = _t('Send verification code')) {
        let remaining = seconds;
        button.disabled = true;

        const timer = setInterval(() => {
            button.textContent = _t('Resend in %s seconds', remaining);
            remaining--;

            if (remaining < 0) {
                clearInterval(timer);
                button.disabled = false;
                button.textContent = originalText;
            }
        }, 1000);

        return timer;
    }
}
