/**
 * Apollo 登录页面 OWL 组件
 */

import { Component, useState, useRef, onMounted, xml } from "@odoo/owl";
import { useService } from "@web/core/utils/hooks";
import { _t } from "@web/core/l10n/translation";
import { ApolloAPI, ApolloStorage, ApolloUtils } from "./apollo_api";

export class ApolloLoginPage extends Component {
    static template = xml`
        <div class="apollo-card">
            <div class="apollo-card-header">
                <h2>用户登录</h2>
                <p>登录您的Apollo账户</p>
            </div>
            <div class="apollo-card-body">
                <!-- 登录方式选择 -->
                <div class="apollo-login-tabs">
                    <button class="apollo-tab-btn" 
                            t-att-class="isActiveTab('password') ? 'active' : ''"
                            t-on-click="() => this.switchTab('password')">密码登录</button>
                    <button class="apollo-tab-btn" 
                            t-att-class="isActiveTab('sms') ? 'active' : ''"
                            t-on-click="() => this.switchTab('sms')">短信登录</button>
                </div>

                <!-- 密码登录表单 -->
                <div t-if="isActiveTab('password')" class="apollo-tab-content active">
                    <form class="apollo-form" t-ref="passwordForm" t-on-submit="onPasswordLogin">
                        <div class="apollo-form-group">
                            <label for="login-phone">手机号或邮箱</label>
                            <input type="text" id="login-phone" name="login" 
                                   placeholder="请输入手机号或邮箱" required=""
                                   t-att-disabled="isLoading()"/>
                        </div>
                        <div class="apollo-form-group">
                            <label for="login-password">密码</label>
                            <input type="password" id="login-password" name="password" 
                                   placeholder="请输入密码" required=""
                                   t-att-disabled="isLoading()"/>
                        </div>
                        <button type="submit" class="apollo-btn apollo-btn-primary"
                                t-att-disabled="isLoading()">
                            <t t-if="isLoading()">登录中...</t>
                            <t t-else="">登录</t>
                        </button>
                    </form>
                </div>

                <!-- 短信登录表单 -->
                <div t-if="isActiveTab('sms')" class="apollo-tab-content active">
                    <form class="apollo-form" t-ref="smsForm" t-on-submit="onSmsLogin">
                        <div class="apollo-form-group">
                            <label for="sms-phone">手机号</label>
                            <input type="tel" id="sms-phone" name="phone" 
                                   placeholder="请输入手机号" required=""
                                   t-att-disabled="isLoading()"/>
                        </div>
                        <div class="apollo-form-group apollo-sms-group">
                            <label for="sms-code">验证码</label>
                            <div class="apollo-sms-input">
                                <input type="text" id="sms-code" name="sms_code" 
                                       placeholder="请输入验证码" required=""
                                       t-att-disabled="isLoading()"/>
                                <button type="button" class="apollo-btn apollo-btn-secondary"
                                        t-on-click="onSendLoginSms"
                                        t-att-disabled="isLoading()">发送验证码</button>
                            </div>
                        </div>
                        <button type="submit" class="apollo-btn apollo-btn-primary"
                                t-att-disabled="isLoading()">
                            <t t-if="isLoading()">登录中...</t>
                            <t t-else="">登录</t>
                        </button>
                    </form>
                </div>

                <div class="apollo-card-footer">
                    <p>还没有账户？<a href="/apollo/register">立即注册</a></p>
                </div>
            </div>
        </div>
    `;

    setup() {
        this.notification = useService("notification");

        this.state = useState({
            activeTab: 'password',
            loading: false,
            formData: {
                login: '',
                password: '',
                phone: '',
                smsCode: ''
            }
        });

        this.passwordFormRef = useRef("passwordForm");
        this.smsFormRef = useRef("smsForm");

        onMounted(() => {
            // 检查是否已登录
            if (ApolloStorage.getToken()) {
                window.location.href = '/apollo/profile';
            }
        });
    }

    /**
     * 切换登录方式标签
     * @param {string} tab 
     */
    switchTab(tab) {
        this.state.activeTab = tab;
    }

    /**
     * 处理密码登录
     * @param {Event} ev 
     */
    async onPasswordLogin(ev) {
        ev.preventDefault();

        const formData = new FormData(ev.target);
        const login = formData.get('login')?.trim();
        const password = formData.get('password');

        if (!login || !password) {
            this.notification.add(_t('Please fill in complete information'), {
                type: 'danger'
            });
            return;
        }

        this.state.loading = true;

        try {
            const result = await ApolloAPI.loginWithPassword(login, password);

            if (result.data?.token) {
                ApolloStorage.saveToken(result.data.token);
                this.notification.add(_t('Login successful!'), {
                    type: 'success'
                });

                setTimeout(() => {
                    window.location.href = '/apollo/profile';
                }, 1000);
            }
        } catch (error) {
            this.notification.add(
                error.message || _t('Login failed, please check username and password'),
                { type: 'danger' }
            );
        } finally {
            this.state.loading = false;
        }
    }

    /**
     * 处理短信验证码登录
     * @param {Event} ev 
     */
    async onSmsLogin(ev) {
        ev.preventDefault();

        const formData = new FormData(ev.target);
        const phone = formData.get('phone')?.trim();
        const smsCode = formData.get('sms_code');

        if (!phone || !smsCode) {
            this.notification.add(_t('Please fill in complete information'), {
                type: 'danger'
            });
            return;
        }

        this.state.loading = true;

        try {
            const result = await ApolloAPI.loginWithSms(phone, smsCode);

            if (result.data?.token) {
                ApolloStorage.saveToken(result.data.token);
                this.notification.add(_t('Login successful!'), {
                    type: 'success'
                });

                setTimeout(() => {
                    window.location.href = '/apollo/profile';
                }, 1000);
            }
        } catch (error) {
            this.notification.add(
                error.message || _t('Login failed, please check verification code'),
                { type: 'danger' }
            );
        } finally {
            this.state.loading = false;
        }
    }

    /**
     * 发送登录短信验证码
     * @param {Event} ev 
     */
    async onSendLoginSms(ev) {
        ev.preventDefault();

        const phoneInput = this.smsFormRef.el?.querySelector('input[name="phone"]');
        const phone = phoneInput?.value?.trim();

        if (!phone) {
            this.notification.add(_t('Please enter phone number'), {
                type: 'danger'
            });
            return;
        }

        const button = ev.target;
        const originalText = button.textContent;

        try {
            button.disabled = true;
            button.textContent = _t('Sending...');

            await ApolloAPI.sendLoginSms(phone);

            this.notification.add(_t('Verification code sent successfully'), {
                type: 'success'
            });

            // 开始倒计时
            ApolloUtils.startCountdown(button, 60, originalText);

        } catch (error) {
            this.notification.add(
                error.message || _t('Failed to send verification code'),
                { type: 'danger' }
            );
            button.disabled = false;
            button.textContent = originalText;
        }
    }

    /**
     * 获取标签是否激活
     * @param {string} tab 
     * @returns {boolean}
     */
    isActiveTab(tab) {
        return this.state.activeTab === tab;
    }

    /**
     * 获取表单是否正在加载
     * @returns {boolean}
     */
    isLoading() {
        return this.state.loading;
    }
}
