/**
 * Apollo 用户信息页面 OWL 组件
 */

import { Component, useState, onMounted, xml } from "@odoo/owl";
import { useService } from "@web/core/utils/hooks";
import { _t } from "@web/core/l10n/translation";
import { ApolloAPI, ApolloStorage, ApolloUtils } from "./apollo_api";

export class ApolloProfilePage extends Component {
    static template = xml`
        <!-- 加载中状态 -->
        <div t-if="isLoading()" class="apollo-loading">
            <div class="apollo-spinner"></div>
            <p>正在加载用户信息...</p>
        </div>

        <!-- 用户信息展示 -->
        <div t-if="hasUserData()" class="apollo-profile-content">
            <!-- 用户基本信息卡片 -->
            <div class="apollo-card apollo-user-card">
                <div class="apollo-card-header">
                    <h2>个人信息</h2>
                    <button class="apollo-btn apollo-btn-secondary" t-on-click="onLogout">退出登录</button>
                </div>
                <div class="apollo-card-body">
                    <div class="apollo-user-info">
                        <div class="apollo-avatar">
                            <div class="apollo-avatar-placeholder" t-esc="getUserAvatarInitials()"></div>
                        </div>
                        <div class="apollo-user-details">
                            <h3 t-esc="getUserName()"></h3>
                            <p t-esc="getUserPhone()"></p>
                            <p t-esc="getUserEmail()"></p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- 组织信息卡片 -->
            <div class="apollo-card apollo-org-card">
                <div class="apollo-card-header">
                    <h2>我的组织</h2>
                    <span class="apollo-badge" t-esc="getOrganizationCount()"></span>
                </div>
                <div class="apollo-card-body">
                    <!-- 没有加入任何组织 -->
                    <div t-if="!hasOrganizations()" class="apollo-empty-state">
                        <p>您还没有加入任何组织</p>
                        <small>通过邀请码可以加入组织</small>
                    </div>
                    
                    <!-- 组织列表 -->
                    <div t-if="hasOrganizations()" class="apollo-org-list">
                        <div t-foreach="getOrganizations()" t-as="org" t-key="org.id" class="apollo-org-item">
                            <div class="apollo-org-info">
                                <div class="apollo-org-logo">
                                    <div class="apollo-org-logo-placeholder" t-esc="getOrgInitials(org)"></div>
                                </div>
                                <div class="apollo-org-details">
                                    <h4 t-esc="org.name"></h4>
                                    <div class="apollo-org-meta">
                                        <span class="apollo-role-badge" t-att-class="getRoleBadgeClass(org)" t-esc="getRoleDisplayName(org)"></span>
                                        <span class="apollo-join-date">加入于 <t t-esc="formatJoinDate(org)"/></span>
                                    </div>
                                    <p class="apollo-org-id"><strong>组织ID:</strong> <span t-esc="org.id"></span></p>
                                </div>
                                <div class="apollo-org-actions">
                                    <button t-if="org.member_info.is_owner" class="apollo-btn apollo-btn-outline" disabled="">组织所有者</button>
                                    <button t-if="!org.member_info.is_owner" class="apollo-btn apollo-btn-outline" disabled="">组织成员</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- JWT Token 信息（调试用，生产环境可隐藏） -->
            <div class="apollo-card apollo-token-card">
                <div class="apollo-card-header">
                    <h2>令牌信息</h2>
                    <small>（调试信息）</small>
                </div>
                <div class="apollo-card-body">
                    <div class="apollo-token-info">
                        <p><strong>Token状态:</strong> 
                           <span class="apollo-status-badge" 
                                 t-att-class="getTokenStatus().class"
                                 t-esc="getTokenStatus().text"></span></p>
                        <p><strong>过期时间:</strong> <span t-esc="getTokenExpiry()"></span></p>
                        <div class="apollo-form-group">
                            <label>JWT Token:</label>
                            <textarea readonly="" class="apollo-token-display" t-esc="getJwtToken()"></textarea>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- 错误状态 -->
        <div t-if="hasError()" class="apollo-error-state">
            <div class="apollo-card">
                <div class="apollo-card-body">
                    <h3>无法获取用户信息</h3>
                    <p t-esc="getErrorMessage()"></p>
                    <a href="/apollo/login" class="apollo-btn apollo-btn-primary">返回登录</a>
                </div>
            </div>
        </div>
    `;

    setup() {
        this.notification = useService("notification");

        this.state = useState({
            loading: true,
            error: false,
            errorMessage: '',
            userData: null,
            organizationData: null
        });

        onMounted(() => {
            this.loadUserProfile();
        });
    }

    /**
     * 加载用户档案信息
     */
    async loadUserProfile() {
        const token = ApolloStorage.getToken();

        if (!token) {
            window.location.href = '/apollo/login';
            return;
        }

        this.state.loading = true;
        this.state.error = false;

        try {
            // 并行获取用户基本信息和组织信息
            const [profileResult, organizationsResult] = await Promise.all([
                ApolloAPI.getUserProfile(token),
                ApolloAPI.getUserOrganizations(token)
            ]);

            if (profileResult.data) {
                this.state.userData = profileResult.data;
                ApolloStorage.saveUserInfo(profileResult.data);
            } else {
                throw new Error(_t('Unable to get user information'));
            }

            if (organizationsResult.data) {
                this.state.organizationData = organizationsResult.data;
            } else {
                // 组织信息是可选的，可能没有加入任何组织
                this.state.organizationData = { organizations: [], total_count: 0, has_owner_role: false };
            }

            this.state.loading = false;
        } catch (error) {
            console.error('Apollo Profile: Load profile error:', error);
            this.state.loading = false;

            // 如果是用户不存在错误，清除token并重定向到登录页面
            if (error.message && (
                error.message.includes('用户不存在') ||
                error.message.includes('User not found') ||
                error.message.includes('Invalid token') ||
                error.message.includes('Token expired')
            )) {
                console.log('Apollo Profile: User not found or invalid token, redirecting to login');
                ApolloStorage.clear();
                window.location.href = '/apollo/login';
                return;
            }

            this.state.error = true;
            this.state.errorMessage = error.message || _t('Failed to get user information');
        }
    }

    /**
     * 处理退出登录
     */
    onLogout() {
        if (confirm(_t('Are you sure you want to logout?'))) {
            ApolloStorage.clear();
            this.notification.add(_t('Logged out'), {
                type: 'info'
            });

            setTimeout(() => {
                window.location.href = '/apollo/login';
            }, 1000);
        }
    }

    /**
     * 获取用户头像字母
     * @returns {string}
     */
    getUserAvatarInitials() {
        return this.state.userData?.name
            ? ApolloUtils.getInitials(this.state.userData.name)
            : '?';
    }

    /**
     * 获取用户姓名
     * @returns {string}
     */
    getUserName() {
        return this.state.userData?.name || _t('Not set');
    }

    /**
     * 获取用户手机号
     * @returns {string}
     */
    getUserPhone() {
        return this.state.userData?.mobile || _t('Not bound');
    }

    /**
     * 获取用户邮箱
     * @returns {string}
     */
    getUserEmail() {
        return this.state.userData?.email || _t('Not bound');
    }

    /**
     * 检查是否有组织
     * @returns {boolean}
     */
    hasOrganizations() {
        return this.state.organizationData?.organizations?.length > 0;
    }

    /**
     * 获取组织数量
     * @returns {string}
     */
    getOrganizationCount() {
        const count = this.state.organizationData?.organizations?.length || 0;
        return count.toString();
    }

    /**
     * 获取所有组织
     * @returns {Array}
     */
    getOrganizations() {
        return this.state.organizationData?.organizations || [];
    }

    /**
     * 获取组织Logo首字母
     * @param {Object} org 组织对象
     * @returns {string}
     */
    getOrgInitials(org) {
        return org?.name ? org.name[0].toUpperCase() : 'O';
    }

    /**
     * 获取角色显示名称
     * @param {Object} org 组织对象
     * @returns {string}
     */
    getRoleDisplayName(org) {
        if (org?.member_info?.is_owner) {
            return _t('Owner');
        }
        return _t('Member');
    }

    /**
     * 获取角色徽章样式类
     * @param {Object} org 组织对象
     * @returns {string}
     */
    getRoleBadgeClass(org) {
        if (org?.member_info?.is_owner) {
            return 'apollo-role-owner';
        }
        return 'apollo-role-member';
    }

    /**
     * 格式化加入日期
     * @param {Object} org 组织对象
     * @returns {string}
     */
    formatJoinDate(org) {
        return org?.member_info?.join_date
            ? ApolloUtils.formatDate(org.member_info.join_date)
            : '-';
    }

    /**
     * 获取JWT令牌
     * @returns {string}
     */
    getJwtToken() {
        return ApolloStorage.getToken() || '';
    }

    /**
     * 获取令牌状态
     * @returns {Object}
     */
    getTokenStatus() {
        const token = ApolloStorage.getToken();
        if (!token) {
            return {
                text: _t('Invalid'),
                class: 'invalid'
            };
        }

        const expiry = ApolloUtils.parseTokenExpiry(token);
        if (expiry && expiry < new Date()) {
            return {
                text: _t('Expired'),
                class: 'invalid'
            };
        }

        return {
            text: _t('Valid'),
            class: 'valid'
        };
    }

    /**
     * 获取令牌过期时间
     * @returns {string}
     */
    getTokenExpiry() {
        const token = ApolloStorage.getToken();
        if (!token) return '-';

        const expiry = ApolloUtils.parseTokenExpiry(token);
        return expiry ? expiry.toLocaleString('zh-CN') : '-';
    }

    /**
     * 是否正在加载
     * @returns {boolean}
     */
    isLoading() {
        return this.state.loading;
    }

    /**
     * 是否有错误
     * @returns {boolean}
     */
    hasError() {
        return this.state.error;
    }

    /**
     * 获取错误消息
     * @returns {string}
     */
    getErrorMessage() {
        return this.state.errorMessage;
    }

    /**
     * 是否有用户数据
     * @returns {boolean}
     */
    hasUserData() {
        return !this.state.loading && !this.state.error && this.state.userData;
    }
}
