/**
 * Apollo 注册页面 OWL 组件
 */

import { Component, useState, useRef, onMounted, xml } from "@odoo/owl";
import { useService } from "@web/core/utils/hooks";
import { _t } from "@web/core/l10n/translation";
import { ApolloAPI, ApolloStorage, ApolloUtils } from "./apollo_api";

export class ApolloRegisterPage extends Component {
    static template = xml`
        <div class="apollo-card">
            <div class="apollo-card-header">
                <h2>用户注册</h2>
                <p>使用邀请码注册Apollo账户</p>
            </div>
            <div class="apollo-card-body">
                <!-- 步骤指示器 -->
                <div class="apollo-steps">
                    <div class="apollo-step" t-att-class="getStepStatus(1)" data-step="1">
                        <div class="apollo-step-number">1</div>
                        <div class="apollo-step-title">验证邀请码</div>
                    </div>
                    <div class="apollo-step" t-att-class="getStepStatus(2)" data-step="2">
                        <div class="apollo-step-number">2</div>
                        <div class="apollo-step-title">短信验证</div>
                    </div>
                    <div class="apollo-step" t-att-class="getStepStatus(3)" data-step="3">
                        <div class="apollo-step-number">3</div>
                        <div class="apollo-step-title">完成注册</div>
                    </div>
                </div>

                <!-- 步骤1：验证邀请码 -->
                <div t-if="isStepActive(1)" class="apollo-step-content active">
                    <form class="apollo-form" t-ref="step1Form" t-on-submit="onVerifyInvitation">
                        <div class="apollo-form-group">
                            <label for="invitation-code">邀请码</label>
                            <input type="text" id="invitation-code" name="invitation_code" 
                                   t-model="state.formData.invitationCode"
                                   placeholder="请输入12位邀请码" required=""
                                   t-att-disabled="isLoading()"/>
                        </div>
                        <button type="submit" class="apollo-btn apollo-btn-primary"
                                t-att-disabled="isLoading()">
                            <t t-if="isLoading()">验证中...</t>
                            <t t-else="">验证邀请码</t>
                        </button>
                    </form>
                </div>

                <!-- 步骤2：短信验证 -->
                <div t-if="isStepActive(2)" class="apollo-step-content active">
                    <form class="apollo-form" t-ref="step2Form" t-on-submit="onVerifySms">
                        <div class="apollo-form-group">
                            <label>绑定手机号</label>
                            <p class="apollo-readonly-field" t-esc="getFormattedPhone()"></p>
                        </div>
                        <div class="apollo-form-group apollo-sms-group">
                            <label for="verify-sms-code">验证码</label>
                            <div class="apollo-sms-input">
                                <input type="text" id="verify-sms-code" name="sms_code" 
                                       placeholder="请输入验证码" required=""
                                       t-att-disabled="isLoading()"/>
                                <button type="button" class="apollo-btn apollo-btn-secondary" 
                                        t-on-click="onSendInvitationSms"
                                        t-att-disabled="isLoading()">发送验证码</button>
                            </div>
                            <small class="apollo-form-hint">
                                💡 温馨提示：如果该手机号已注册，验证通过后将自动登录
                            </small>
                        </div>
                        <button type="submit" class="apollo-btn apollo-btn-primary"
                                t-att-disabled="isLoading()">
                            <t t-if="isLoading()">验证中...</t>
                            <t t-else="">验证短信</t>
                        </button>
                    </form>
                </div>

                <!-- 步骤3：完成注册 -->
                <div t-if="isStepActive(3)" class="apollo-step-content active">
                    <form class="apollo-form" t-ref="step3Form" t-on-submit="onRegister">
                        <div class="apollo-form-group">
                            <label for="user-name">姓名</label>
                            <input type="text" id="user-name" name="name" 
                                   placeholder="请输入您的姓名" required=""
                                   t-att-disabled="isLoading()"/>
                        </div>
                        <div class="apollo-form-group">
                            <label for="user-password">密码</label>
                            <input type="password" id="user-password" name="password" 
                                   placeholder="请设置密码" required=""
                                   t-att-disabled="isLoading()"/>
                        </div>
                        <div class="apollo-form-group">
                            <label for="confirm-password">确认密码</label>
                            <input type="password" id="confirm-password" name="confirm_password" 
                                   placeholder="请再次输入密码" required=""
                                   t-att-disabled="isLoading()"/>
                        </div>
                        <button type="submit" class="apollo-btn apollo-btn-primary"
                                t-att-disabled="isLoading()">
                            <t t-if="isLoading()">注册中...</t>
                            <t t-else="">完成注册</t>
                        </button>
                    </form>
                </div>

                <div class="apollo-card-footer">
                    <p>已有账户？<a href="/apollo/login">立即登录</a></p>
                </div>
            </div>
        </div>
    `;

    setup() {
        this.notification = useService("notification");

        this.state = useState({
            currentStep: 1,
            loading: false,
            invitationData: null,
            registrationToken: null,
            formData: {
                invitationCode: '',
                smsCode: '',
                name: '',
                password: '',
                confirmPassword: ''
            }
        });

        this.step1FormRef = useRef("step1Form");
        this.step2FormRef = useRef("step2Form");
        this.step3FormRef = useRef("step3Form");

        onMounted(() => {
            // 检查是否已登录
            if (ApolloStorage.getToken()) {
                window.location.href = '/apollo/profile';
                return;
            }

            // 检查URL参数中的邀请码
            const urlParams = new URLSearchParams(window.location.search);
            const invitationCode = urlParams.get('invitation_code');
            if (invitationCode) {
                this.state.formData.invitationCode = invitationCode;
            }
        });
    }

    /**
     * 处理邀请码验证
     * @param {Event} ev 
     */
    async onVerifyInvitation(ev) {
        ev.preventDefault();

        const formData = new FormData(ev.target);
        const invitationCode = formData.get('invitation_code')?.trim();

        if (!invitationCode) {
            this.notification.add(_t('Please enter invitation code'), {
                type: 'danger'
            });
            return;
        }

        this.state.loading = true;

        try {
            const result = await ApolloAPI.getInvitationInfo(invitationCode);

            if (result.data) {
                this.state.invitationData = result.data;
                this.notification.add(_t('Invitation code verified successfully'), {
                    type: 'success'
                });

                setTimeout(() => {
                    this.goToStep(2);
                }, 1000);
            }
        } catch (error) {
            this.notification.add(
                error.message || _t('Invalid invitation code'),
                { type: 'danger' }
            );
        } finally {
            this.state.loading = false;
        }
    }

    /**
     * 发送邀请注册短信验证码
     * @param {Event} ev 
     */
    async onSendInvitationSms(ev) {
        ev.preventDefault();

        if (!this.state.invitationData) {
            this.notification.add(_t('Please verify invitation code first'), {
                type: 'danger'
            });
            return;
        }

        const button = ev.target;
        const originalText = button.textContent;

        try {
            button.disabled = true;
            button.textContent = _t('Sending...');

            await ApolloAPI.sendInvitationSms(this.state.invitationData.invitation_code);

            this.notification.add(_t('Verification code sent successfully'), {
                type: 'success'
            });

            // 开始倒计时
            ApolloUtils.startCountdown(button, 60, originalText);

        } catch (error) {
            this.notification.add(
                error.message || _t('Failed to send verification code'),
                { type: 'danger' }
            );
            button.disabled = false;
            button.textContent = originalText;
        }
    }

    /**
     * 处理短信验证
     * @param {Event} ev 
     */
    async onVerifySms(ev) {
        ev.preventDefault();

        const formData = new FormData(ev.target);
        const smsCode = formData.get('sms_code');

        if (!smsCode) {
            this.notification.add(_t('Please enter verification code'), {
                type: 'danger'
            });
            return;
        }

        if (!this.state.invitationData) {
            this.notification.add(_t('Please verify invitation code first'), {
                type: 'danger'
            });
            return;
        }

        this.state.loading = true;

        try {
            const result = await ApolloAPI.verifyInvitationSms(
                this.state.invitationData.invitation_code,
                smsCode
            );

            if (result.data?.token) {
                // 检查用户是否已存在
                if (result.data.user_exists && result.data.status === 'login_success') {
                    // 用户已存在，直接登录成功
                    ApolloStorage.saveToken(result.data.token);

                    // 保存用户信息
                    if (result.data.user) {
                        ApolloStorage.saveUserInfo(result.data.user);
                    }

                    this.notification.add(_t('Welcome back! Logged in successfully'), {
                        type: 'success'
                    });

                    setTimeout(() => {
                        window.location.href = '/apollo/profile';
                    }, 1500);

                } else if (!result.data.user_exists && result.data.status === 'registration_pending') {
                    // 用户不存在，需要继续注册流程
                    this.state.registrationToken = result.data.token;
                    this.notification.add(_t('SMS verification successful, please complete registration'), {
                        type: 'success'
                    });

                    setTimeout(() => {
                        this.goToStep(3);
                    }, 1000);
                } else {
                    // 其他情况，继续原有流程
                    this.state.registrationToken = result.data.token;
                    this.notification.add(_t('SMS verification successful'), {
                        type: 'success'
                    });

                    setTimeout(() => {
                        this.goToStep(3);
                    }, 1000);
                }
            }
        } catch (error) {
            this.notification.add(
                error.message || _t('Verification code error'),
                { type: 'danger' }
            );
        } finally {
            this.state.loading = false;
        }
    }

    /**
     * 处理用户注册
     * @param {Event} ev 
     */
    async onRegister(ev) {
        ev.preventDefault();

        const formData = new FormData(ev.target);
        const name = formData.get('name')?.trim();
        const password = formData.get('password');
        const confirmPassword = formData.get('confirm_password');

        if (!name || !password || !confirmPassword) {
            this.notification.add(_t('Please fill in complete information'), {
                type: 'danger'
            });
            return;
        }

        if (password !== confirmPassword) {
            this.notification.add(_t('Password confirmation does not match'), {
                type: 'danger'
            });
            return;
        }

        if (password.length < 6) {
            this.notification.add(_t('Password must be at least 6 characters'), {
                type: 'danger'
            });
            return;
        }

        if (!this.state.registrationToken) {
            this.notification.add(_t('Invalid registration token, please verify again'), {
                type: 'danger'
            });
            return;
        }

        this.state.loading = true;

        try {
            const result = await ApolloAPI.register(
                this.state.registrationToken,
                name,
                password
            );

            if (result.data?.token) {
                ApolloStorage.saveToken(result.data.token);
                this.notification.add(_t('Registration successful! Redirecting...'), {
                    type: 'success'
                });

                setTimeout(() => {
                    window.location.href = '/apollo/profile';
                }, 2000);
            }
        } catch (error) {
            this.notification.add(
                error.message || _t('Registration failed, please try again later'),
                { type: 'danger' }
            );
        } finally {
            this.state.loading = false;
        }
    }

    /**
     * 跳转到指定步骤
     * @param {number} step 
     */
    goToStep(step) {
        this.state.currentStep = step;
    }

    /**
     * 获取步骤状态
     * @param {number} step 
     * @returns {string}
     */
    getStepStatus(step) {
        if (step < this.state.currentStep) {
            return 'completed';
        } else if (step === this.state.currentStep) {
            return 'active';
        }
        return '';
    }

    /**
     * 获取步骤内容是否显示
     * @param {number} step 
     * @returns {boolean}
     */
    isStepActive(step) {
        return this.state.currentStep === step;
    }

    /**
     * 获取格式化的手机号
     * @returns {string}
     */
    getFormattedPhone() {
        return this.state.invitationData?.phone
            ? ApolloUtils.formatPhone(this.state.invitationData.phone)
            : '';
    }

    /**
     * 获取表单是否正在加载
     * @returns {boolean}
     */
    isLoading() {
        return this.state.loading;
    }
}
