# Apollo Package Management 使用示例

## 基础使用示例

### 1. 检查用户功能访问权限

```python
# 在控制器或业务逻辑中检查用户是否有访问学院功能的权限
permission_service = request.env['apollo.package.permission.service']

# 检查当前用户是否有学院访问权限
access_result = permission_service.check_feature_access(
    user_id=request.env.user.id,
    module_code='academy'
)

if access_result['access']:
    # 用户有权限，允许访问
    return request.render('academy.main_page')
else:
    # 用户无权限，显示升级提示
    return request.render('apollo_package.upgrade_required', {
        'message': access_result['message'],
        'organizations': access_result['organizations']
    })
```

### 2. 消费额度

```python
# 在使用AI对话功能时消费额度
permission_service = request.env['apollo.package.permission.service']

# 检查并消费Hanna AI对话额度
consume_result = permission_service.consume_user_quota(
    user_id=request.env.user.id,
    quota_code='hanna_conversations',
    amount=1
)

if consume_result['success']:
    # 额度消费成功，执行AI对话
    ai_response = hanna_ai.chat(user_message)
    return {'status': 'success', 'response': ai_response}
else:
    # 额度不足，返回错误信息
    return {'status': 'error', 'message': consume_result['message']}
```

### 3. 批量权限检查

```python
# 检查用户的多个权限
permission_service = request.env['apollo.package.permission.service']

permissions_to_check = [
    {'type': 'feature', 'code': 'academy'},
    {'type': 'feature', 'code': 'tools'},
    {'type': 'quota', 'code': 'hanna_conversations', 'amount': 1},
    {'type': 'quota', 'code': 'bid_optimization', 'amount': 5},
]

results = permission_service.check_bulk_permissions(
    user_id=request.env.user.id,
    permission_list=permissions_to_check
)

# 根据结果显示可用功能
available_features = []
for code, result in results.items():
    if result['access']:
        available_features.append(code)

return {'available_features': available_features}
```

## 管理员操作示例

### 1. 为组织创建套餐订阅

```python
# 管理员为组织分配企业套餐
subscription = request.env['apollo.organization.package.subscription'].sudo().create({
    'organization_id': organization.id,
    'package_id': enterprise_package.id,
    'start_date': fields.Date.today(),
    'end_date': fields.Date.today() + timedelta(weeks=12),
    'status': 'active',
    'created_by': request.env.user.id,
})

# 自动创建额度使用记录
quota_service = request.env['apollo.package.quota.service']
effective_quotas = quota_service.get_organization_effective_quotas(organization.id)

for quota_code, quota_info in effective_quotas.items():
    quota_definition = request.env['apollo.quota.definition'].search([
        ('quota_code', '=', quota_code)
    ], limit=1)
    
    if quota_definition:
        request.env['apollo.organization.quota.usage'].create({
            'organization_id': organization.id,
            'subscription_id': subscription.id,
            'quota_definition_id': quota_definition.id,
            'used_amount': 0,
        })
```

### 2. 调整组织额度

```python
# 为VIP客户增加AI对话次数
adjustment = request.env['apollo.organization.quota.adjustment'].sudo().create({
    'organization_id': vip_organization.id,
    'quota_definition_id': hanna_conversations_quota.id,
    'adjustment_type': 'add',
    'adjustment_value': 5000,
    'reason': 'VIP客户额外福利',
    'effective_date': fields.Date.today(),
    'created_by': request.env.user.id,
})

# 临时性额度提升（促销活动）
temp_adjustment = request.env['apollo.organization.quota.adjustment'].sudo().create({
    'organization_id': organization.id,
    'quota_definition_id': cyber_minigun_quota.id,
    'adjustment_type': 'multiply',
    'adjustment_value': 2,
    'reason': '双十一促销活动',
    'effective_date': fields.Date.today(),
    'expiry_date': fields.Date.today() + timedelta(days=30),
    'created_by': request.env.user.id,
})
```

### 3. 延长订阅时间

```python
# 为客户延长30天使用期限
extension = request.env['apollo.organization.subscription.extension'].sudo().create({
    'subscription_id': subscription.id,
    'extension_days': 30,
    'original_end_date': subscription.final_end_date,
    'new_end_date': subscription.final_end_date + timedelta(days=30),
    'reason': '服务补偿',
    'created_by': request.env.user.id,
})
```

## 高级使用场景

### 1. 自定义权限装饰器

```python
from functools import wraps
from odoo.exceptions import AccessDenied

def require_quota(quota_code, amount=1):
    """装饰器：要求特定额度"""
    def decorator(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):
            permission_service = self.env['apollo.package.permission.service']
            permission_service.require_quota_available(
                quota_code, amount, self.env.user.id
            )
            return func(self, *args, **kwargs)
        return wrapper
    return decorator

def require_feature(module_code):
    """装饰器：要求功能访问权限"""
    def decorator(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):
            permission_service = self.env['apollo.package.permission.service']
            permission_service.require_feature_access(
                module_code, self.env.user.id
            )
            return func(self, *args, **kwargs)
        return wrapper
    return decorator

# 使用示例
class AcademyController(http.Controller):
    
    @require_feature('academy')
    @http.route('/academy/courses', auth='user')
    def list_courses(self):
        # 只有有学院权限的用户才能访问
        return request.render('academy.courses')
    
    @require_quota('hanna_conversations', 1)
    @http.route('/academy/ai_tutor', auth='user', methods=['POST'])
    def ai_tutor_chat(self):
        # 自动检查和消费AI对话额度
        return self._process_ai_chat()
```

### 2. 额度监控和告警

```python
class QuotaMonitor(models.Model):
    _name = 'apollo.quota.monitor'
    
    @api.model
    def check_quota_alerts(self):
        """检查额度使用告警"""
        usage_records = self.env['apollo.organization.quota.usage'].search([
            ('usage_percentage', '>=', 80),
            ('is_unlimited', '=', False)
        ])
        
        for usage in usage_records:
            if usage.usage_percentage >= 95:
                self._send_critical_alert(usage)
            elif usage.usage_percentage >= 90:
                self._send_warning_alert(usage)
    
    def _send_critical_alert(self, usage):
        """发送严重告警"""
        self.env['mail.mail'].create({
            'subject': f'额度严重不足告警 - {usage.organization_name}',
            'body_html': f'''
                <p>组织 <strong>{usage.organization_name}</strong> 的额度使用已达到 {usage.usage_percentage:.1f}%</p>
                <p>额度类型：{usage.quota_name}</p>
                <p>已使用：{usage.used_amount} / {usage.quota_limit}</p>
                <p>请及时处理！</p>
            ''',
            'email_to': 'admin@apollo.com',
        }).send()
    
    @api.model
    def _cron_quota_monitor(self):
        """定时任务：额度监控"""
        self.check_quota_alerts()
```

### 3. 额度使用统计

```python
class QuotaAnalytics(models.Model):
    _name = 'apollo.quota.analytics'
    
    @api.model
    def get_usage_statistics(self, organization_id=None, date_from=None, date_to=None):
        """获取额度使用统计"""
        domain = []
        if organization_id:
            domain.append(('organization_id', '=', organization_id))
        if date_from:
            domain.append(('last_updated', '>=', date_from))
        if date_to:
            domain.append(('last_updated', '<=', date_to))
        
        usage_records = self.env['apollo.organization.quota.usage'].search(domain)
        
        statistics = {
            'total_organizations': len(usage_records.mapped('organization_id')),
            'total_quotas': len(usage_records),
            'by_category': {},
            'by_organization': {},
            'top_consumers': [],
        }
        
        # 按分类统计
        for category in ['accounts', 'conversations', 'actions', 'features']:
            category_records = usage_records.filtered(
                lambda r: r.quota_category == category
            )
            statistics['by_category'][category] = {
                'total_used': sum(category_records.mapped('used_amount')),
                'total_limit': sum(category_records.mapped('quota_limit')),
                'avg_usage_percentage': sum(category_records.mapped('usage_percentage')) / len(category_records) if category_records else 0,
            }
        
        # 按组织统计
        for org in usage_records.mapped('organization_id'):
            org_records = usage_records.filtered(lambda r: r.organization_id == org)
            statistics['by_organization'][org.name] = {
                'total_quotas': len(org_records),
                'avg_usage_percentage': sum(org_records.mapped('usage_percentage')) / len(org_records),
                'over_limit_count': len(org_records.filtered('is_over_limit')),
            }
        
        return statistics
```

## 前端集成示例

### 1. JavaScript权限检查

```javascript
// 前端权限检查
class PermissionManager {
    constructor() {
        this.permissions = {};
    }
    
    async checkPermission(moduleCode) {
        if (this.permissions[moduleCode] === undefined) {
            const result = await this._rpc({
                model: 'apollo.package.permission.service',
                method: 'check_feature_access',
                args: [odoo.session.user_id, moduleCode],
            });
            this.permissions[moduleCode] = result.access;
        }
        return this.permissions[moduleCode];
    }
    
    async checkQuota(quotaCode, amount = 1) {
        const result = await this._rpc({
            model: 'apollo.package.permission.service',
            method: 'check_quota_permission',
            args: [odoo.session.user_id, quotaCode, amount],
        });
        return result;
    }
    
    async consumeQuota(quotaCode, amount = 1) {
        const result = await this._rpc({
            model: 'apollo.package.permission.service',
            method: 'consume_user_quota',
            args: [odoo.session.user_id, quotaCode, amount],
        });
        return result;
    }
}

// 使用示例
const permissionManager = new PermissionManager();

// 检查学院权限
if (await permissionManager.checkPermission('academy')) {
    $('#academy-menu').show();
} else {
    $('#academy-menu').hide();
}

// 使用AI对话前检查额度
$('#ai-chat-button').click(async function() {
    const quotaCheck = await permissionManager.checkQuota('hanna_conversations', 1);
    if (quotaCheck.access) {
        const consumeResult = await permissionManager.consumeQuota('hanna_conversations', 1);
        if (consumeResult.success) {
            // 执行AI对话
            startAIChat();
        }
    } else {
        showUpgradeDialog(quotaCheck.message);
    }
});
```

### 2. 额度使用进度条

```xml
<!-- 额度使用进度条模板 -->
<template id="quota_usage_widget">
    <div class="quota-usage-widget">
        <div class="quota-header">
            <span class="quota-name" t-esc="quota.name"/>
            <span class="quota-usage" t-esc="quota.used + ' / ' + (quota.unlimited ? '∞' : quota.limit)"/>
        </div>
        <div class="progress" t-if="not quota.unlimited">
            <div class="progress-bar" 
                 t-attf-class="progress-bar-#{quota.status}"
                 t-attf-style="width: #{quota.percentage}%">
            </div>
        </div>
        <div class="quota-status" t-if="quota.unlimited">
            <span class="badge badge-success">无限制</span>
        </div>
    </div>
</template>
```

这些示例展示了Apollo Package Management系统的各种使用场景，从基础的权限检查到高级的监控和分析功能。系统的设计使得权限管理变得简单而灵活，同时提供了强大的扩展能力。
