# -*- coding: utf-8 -*-

from odoo import models, fields, api


class FeatureModule(models.Model):
    """功能模块表 - 定义系统中的功能模块"""
    _name = 'apollo.feature.module'
    _description = '功能模块'
    _order = 'sequence, id'
    _rec_name = 'module_name'

    module_code = fields.Char(
        string='模块编码',
        required=True,
        size=50,
        help='模块的唯一标识符'
    )
    module_name = fields.Char(
        string='模块名称',
        required=True,
        size=100,
        help='模块的显示名称'
    )
    description = fields.Text(
        string='模块描述',
        help='模块的详细描述'
    )
    is_active = fields.Boolean(
        string='是否激活',
        default=True,
        help='是否激活此模块'
    )
    sequence = fields.Integer(
        string='排序序号',
        default=10,
        help='用于排序的序号'
    )

    # 关联字段
    quota_definition_ids = fields.One2many(
        'apollo.quota.definition',
        'feature_module_id',
        string='相关额度定义'
    )

    _sql_constraints = [
        ('module_code_unique', 'unique(module_code)', '模块编码必须唯一！'),
    ]

    @api.model
    def name_search(self, name='', args=None, operator='ilike', limit=100):
        """支持按编码和名称搜索"""
        if args is None:
            args = []
        domain = args[:]
        if name:
            domain = ['|', ('module_code', operator, name), ('module_name', operator, name)] + domain
        records = self.search(domain, limit=limit, order=self._order)
        return records.name_get()

    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = f"[{record.module_code}] {record.module_name}"
            result.append((record.id, name))
        return result
