# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError


class OrganizationQuotaAdjustment(models.Model):
    """组织额度调整记录表 - 记录组织的个人额度调整"""
    _name = 'apollo.organization.quota.adjustment'
    _description = '组织额度调整记录'
    _order = 'created_date desc, id'
    _rec_name = 'display_name'

    organization_id = fields.Many2one(
        'apollo.organization',
        string='组织',
        required=True,
        ondelete='cascade',
        help='关联的组织'
    )
    subscription_id = fields.Many2one(
        'apollo.organization.package.subscription',
        string='订阅',
        help='关联的订阅，为空表示全局调整'
    )
    quota_definition_id = fields.Many2one(
        'apollo.quota.definition',
        string='额度定义',
        required=True,
        help='要调整的额度定义'
    )
    adjustment_type = fields.Selection([
        ('add', '增加'),
        ('set', '设置'),
        ('multiply', '倍数'),
    ], string='调整类型', required=True, help='额度调整类型')
    
    adjustment_value = fields.Integer(
        string='调整数值',
        required=True,
        default=0,
        help='调整的数值'
    )
    reason = fields.Text(
        string='调整原因',
        help='额度调整的原因'
    )
    effective_date = fields.Date(
        string='生效日期',
        required=True,
        default=fields.Date.today,
        help='调整生效日期'
    )
    expiry_date = fields.Date(
        string='失效日期',
        help='调整失效日期，为空表示永久'
    )
    is_active = fields.Boolean(
        string='是否激活',
        default=True,
        help='是否激活此调整'
    )
    created_by = fields.Many2one(
        'res.users',
        string='创建人',
        required=True,
        default=lambda self: self.env.user,
        help='创建此调整的用户'
    )
    created_date = fields.Datetime(
        string='创建时间',
        default=fields.Datetime.now,
        readonly=True
    )



    # 计算字段
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    is_current_effective = fields.Boolean(
        string='是否当前有效',
        compute='_compute_is_current_effective',
        search='_search_is_current_effective'
    )
    adjustment_description = fields.Char(
        string='调整描述',
        compute='_compute_adjustment_description'
    )

    _sql_constraints = [
        ('adjustment_value_positive', 'check(adjustment_value > 0)', 
         '调整数值必须大于0！'),
    ]

    @api.depends('organization_id.name', 'quota_definition_id.quota_name', 'adjustment_type', 'adjustment_value')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.organization_id and record.quota_definition_id:
                type_dict = dict(record._fields['adjustment_type'].selection)
                type_name = type_dict.get(record.adjustment_type, '')
                record.display_name = f"{record.organization_id.name} - {record.quota_definition_id.quota_name} ({type_name} {record.adjustment_value})"
            else:
                record.display_name = "新调整"

    @api.depends('effective_date', 'expiry_date', 'is_active')
    def _compute_is_current_effective(self):
        """计算是否当前有效"""
        today = fields.Date.today()
        for record in self:
            record.is_current_effective = (
                record.is_active and
                record.effective_date <= today and
                (not record.expiry_date or record.expiry_date >= today)
            )

    @api.depends('adjustment_type', 'adjustment_value', 'quota_definition_id.unit')
    def _compute_adjustment_description(self):
        """计算调整描述"""
        for record in self:
            type_dict = dict(record._fields['adjustment_type'].selection)
            type_name = type_dict.get(record.adjustment_type, '')
            unit = record.quota_definition_id.unit if record.quota_definition_id else 'count'
            
            if record.adjustment_type == 'add':
                record.adjustment_description = f"增加 {record.adjustment_value} {unit}"
            elif record.adjustment_type == 'set':
                record.adjustment_description = f"设置为 {record.adjustment_value} {unit}"
            elif record.adjustment_type == 'multiply':
                record.adjustment_description = f"乘以 {record.adjustment_value} 倍"
            else:
                record.adjustment_description = f"{type_name} {record.adjustment_value}"

    def _search_is_current_effective(self, operator, value):
        """搜索当前有效的额度调整"""
        today = fields.Date.today()
        if operator == '=' and value:
            return [
                ('is_active', '=', True),
                ('effective_date', '<=', today),
                '|',
                ('expiry_date', '=', False),
                ('expiry_date', '>=', today)
            ]
        elif operator == '=' and not value:
            return [
                '|',
                ('is_active', '=', False),
                '|',
                ('effective_date', '>', today),
                ('expiry_date', '<', today)
            ]
        else:
            return []

    @api.constrains('effective_date', 'expiry_date')
    def _check_date_range(self):
        """检查日期范围"""
        for record in self:
            if record.expiry_date and record.effective_date > record.expiry_date:
                raise ValidationError('生效日期不能晚于失效日期！')

    @api.constrains('adjustment_type', 'adjustment_value')
    def _check_adjustment_value(self):
        """检查调整数值"""
        for record in self:
            if record.adjustment_type == 'multiply' and record.adjustment_value <= 0:
                raise ValidationError('倍数调整的数值必须大于0！')
            elif record.adjustment_type in ['add', 'set'] and record.adjustment_value < 0:
                raise ValidationError('增加和设置调整的数值不能为负数！')

    @api.onchange('subscription_id')
    def _onchange_subscription_id(self):
        """当选择订阅时，自动设置组织"""
        if self.subscription_id:
            self.organization_id = self.subscription_id.organization_id

    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = record.adjustment_description
            if record.is_current_effective:
                name += " [有效]"
            elif not record.is_active:
                name += " [已禁用]"
            elif record.expiry_date and record.expiry_date < fields.Date.today():
                name += " [已过期]"
            result.append((record.id, name))
        return result

    def action_toggle_active(self):
        """切换激活状态"""
        for record in self:
            record.is_active = not record.is_active

    def action_extend_expiry(self):
        """延长失效日期"""
        return {
            'name': '延长失效日期',
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.quota.adjustment.extend.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {'default_adjustment_id': self.id},
        }

    @api.model
    def get_effective_adjustments(self, organization_id, quota_code=None, date=None):
        """获取组织的有效额度调整"""
        if date is None:
            date = fields.Date.today()
        
        domain = [
            ('organization_id', '=', organization_id),
            ('is_active', '=', True),
            ('effective_date', '<=', date),
            '|',
            ('expiry_date', '=', False),
            ('expiry_date', '>=', date)
        ]
        
        if quota_code:
            domain.append(('quota_definition_id.quota_code', '=', quota_code))
        
        return self.search(domain)

    @api.model
    def _cron_deactivate_expired_adjustments(self):
        """定时任务：停用过期的额度调整"""
        today = fields.Date.today()
        expired_adjustments = self.search([
            ('is_active', '=', True),
            ('expiry_date', '<', today)
        ])
        expired_adjustments.write({'is_active': False})
