# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError


class PackagePricingChangeLog(models.Model):
    """套餐定价修改记录表 - 记录所有定价变更历史"""
    _name = 'apollo.package.pricing.change.log'
    _description = '套餐定价修改记录'
    _order = 'change_date desc, id desc'
    _rec_name = 'display_name'

    # 基本信息
    package_id = fields.Many2one(
        'apollo.package',
        string='关联套餐',
        required=True,
        readonly=True,
        help='关联的套餐'
    )
    pricing_id = fields.Many2one(
        'apollo.package.pricing',
        string='定价记录',
        required=True,
        readonly=True,
        help='关联的定价记录'
    )
    
    # 变更信息
    change_type = fields.Selection([
        ('create', '新建'),
        ('update', '修改'),
        ('delete', '删除'),
        ('activate', '激活'),
        ('deactivate', '停用'),
    ], string='变更类型', required=True, readonly=True, help='定价变更类型')
    
    change_date = fields.Datetime(
        string='变更时间',
        required=True,
        readonly=True,
        default=fields.Datetime.now,
        help='变更发生时间'
    )
    
    # 变更前后的值
    old_price = fields.Float(
        string='原价格',
        readonly=True,
        help='变更前的价格'
    )
    new_price = fields.Float(
        string='新价格',
        readonly=True,
        help='变更后的价格'
    )
    old_currency = fields.Selection([
        ('CNY', '人民币'),
        ('USD', '美元'),
        ('EUR', '欧元'),
    ], string='原货币类型', readonly=True, help='变更前的货币类型')
    new_currency = fields.Selection([
        ('CNY', '人民币'),
        ('USD', '美元'),
        ('EUR', '欧元'),
    ], string='新货币类型', readonly=True, help='变更后的货币类型')
    
    old_effective_date = fields.Date(
        string='原生效日期',
        readonly=True,
        help='变更前的生效日期'
    )
    new_effective_date = fields.Date(
        string='新生效日期',
        readonly=True,
        help='变更后的生效日期'
    )
    old_expiry_date = fields.Date(
        string='原失效日期',
        readonly=True,
        help='变更前的失效日期'
    )
    new_expiry_date = fields.Date(
        string='新失效日期',
        readonly=True,
        help='变更后的失效日期'
    )
    

    
    # 操作信息
    user_id = fields.Many2one(
        'res.users',
        string='操作用户',
        required=True,
        readonly=True,
        default=lambda self: self.env.user,
        help='执行变更的用户'
    )
    reason = fields.Text(
        string='变更原因',
        readonly=True,
        help='变更原因说明'
    )
    notes = fields.Text(
        string='备注',
        readonly=True,
        help='额外备注信息'
    )
    

    user_name = fields.Char(
        related='user_id.name',
        string='操作用户名',
        readonly=True,
        store=True
    )
    
    # 计算字段
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    change_summary = fields.Text(
        string='变更摘要',
        compute='_compute_change_summary'
    )
    price_change_amount = fields.Float(
        string='价格变化金额',
        compute='_compute_price_change_amount',
        search='_search_price_change_amount',
        store=True
    )
    price_change_percentage = fields.Float(
        string='价格变化百分比',
        compute='_compute_price_change_percentage',
        search='_search_price_change_percentage',
        store=True
    )
    
    @api.depends('package_id.name', 'change_type', 'change_date')
    def _compute_display_name(self):
        """计算显示名称"""
        type_dict = dict(self._fields['change_type'].selection)
        for record in self:
            change_type_label = type_dict.get(record.change_type, '')
            date_str = record.change_date.strftime('%Y-%m-%d %H:%M') if record.change_date else ''
            package_name = record.package_id.name if record.package_id else '未知套餐'
            record.display_name = f"{package_name} - {change_type_label} ({date_str})"
    
    @api.depends('change_type', 'old_price', 'new_price', 'old_currency', 'new_currency',
                 'old_effective_date', 'new_effective_date', 'old_expiry_date', 'new_expiry_date')
    def _compute_change_summary(self):
        """计算变更摘要"""
        for record in self:
            summary_lines = []
            
            if record.change_type == 'create':
                summary_lines.append(f"新建定价: {record.new_currency} {record.new_price}")
            elif record.change_type == 'delete':
                summary_lines.append(f"删除定价: {record.old_currency} {record.old_price}")
            else:
                # 检查价格变更
                if record.old_price != record.new_price:
                    summary_lines.append(f"价格: {record.old_currency} {record.old_price} → {record.new_currency} {record.new_price}")
                
                # 检查货币变更
                if record.old_currency != record.new_currency:
                    summary_lines.append(f"货币: {record.old_currency} → {record.new_currency}")
                
                # 检查生效日期变更
                if record.old_effective_date != record.new_effective_date:
                    old_date = record.old_effective_date.strftime('%Y-%m-%d') if record.old_effective_date else '未设置'
                    new_date = record.new_effective_date.strftime('%Y-%m-%d') if record.new_effective_date else '未设置'
                    summary_lines.append(f"生效日期: {old_date} → {new_date}")
                
                # 检查失效日期变更
                if record.old_expiry_date != record.new_expiry_date:
                    old_date = record.old_expiry_date.strftime('%Y-%m-%d') if record.old_expiry_date else '永久'
                    new_date = record.new_expiry_date.strftime('%Y-%m-%d') if record.new_expiry_date else '永久'
                    summary_lines.append(f"失效日期: {old_date} → {new_date}")

            
            record.change_summary = '\n'.join(summary_lines) if summary_lines else '无变更'
    
    @api.depends('old_price', 'new_price')
    def _compute_price_change_amount(self):
        """计算价格变化金额"""
        for record in self:
            if record.old_price is not False and record.new_price is not False:
                record.price_change_amount = record.new_price - record.old_price
            else:
                record.price_change_amount = 0.0
    
    @api.depends('old_price', 'new_price')
    def _compute_price_change_percentage(self):
        """计算价格变化百分比"""
        for record in self:
            if record.old_price and record.old_price > 0 and record.new_price is not False:
                record.price_change_percentage = ((record.new_price - record.old_price) / record.old_price) * 100
            else:
                record.price_change_percentage = 0.0
    
    def _search_price_change_amount(self, operator, value):
        """搜索价格变化金额"""
        return [('price_change_amount', operator, value)]
    
    def _search_price_change_percentage(self, operator, value):
        """搜索价格变化百分比"""
        return [('price_change_percentage', operator, value)]
    
    @api.model
    def create_change_log(self, pricing_record, change_type, old_values=None, reason=None, notes=None):
        """创建定价变更记录的工具方法"""
        vals = {
            'package_id': pricing_record.package_id.id,
            'pricing_id': pricing_record.id,
            'change_type': change_type,
            'user_id': self.env.user.id,
            'reason': reason or '',
            'notes': notes or '',
        }
        
        # 设置新值
        vals.update({
            'new_price': pricing_record.price,
            'new_currency': pricing_record.currency,
            'new_effective_date': pricing_record.effective_date,
            'new_expiry_date': pricing_record.expiry_date,
        })
        
        # 设置旧值
        if old_values and change_type in ['update', 'delete']:
            vals.update({
                'old_price': old_values.get('price', pricing_record.price),
                'old_currency': old_values.get('currency', pricing_record.currency),
                'old_effective_date': old_values.get('effective_date', pricing_record.effective_date),
                'old_expiry_date': old_values.get('expiry_date', pricing_record.expiry_date),
            })
        elif change_type == 'create':
            # 新建时旧值为None/False
            vals.update({
                'old_price': 0.0,
                'old_currency': False,
                'old_effective_date': False,
                'old_expiry_date': False,
            })
        
        return self.create(vals)
    
    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = record.display_name
            if record.reason:
                name += f" - {record.reason[:50]}..."
            result.append((record.id, name))
        return result
