# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError, UserError


class PackageQuota(models.Model):
    """套餐额度关联表 - 定义套餐包含的额度"""
    _name = 'apollo.package.quota'
    _description = '套餐额度'
    _order = 'package_id, quota_definition_id'
    _rec_name = 'display_name'

    package_id = fields.Many2one(
        'apollo.package',
        string='关联套餐',
        required=True,
        ondelete='cascade',
        help='关联的套餐'
    )
    quota_definition_id = fields.Many2one(
        'apollo.quota.definition',
        string='额度定义',
        required=True,
        help='关联的额度定义'
    )
    quota_limit = fields.Integer(
        string='额度限制数量',
        default=0,
        help='额度的限制数量'
    )
    is_unlimited = fields.Boolean(
        string='是否无限制',
        default=False,
        help='是否无限制使用'
    )
    is_additional = fields.Boolean(
        string='是否为追加额度',
        default=False,
        help='是否为追加额度（用于扩展套餐）'
    )
    is_active = fields.Boolean(
        string='是否启用',
        default=True,
        help='是否启用此额度'
    )
    is_deleted = fields.Boolean(
        string='是否已删除',
        default=False,
        help='标记额度是否已被删除（软删除）'
    )



    # 计算字段
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    effective_limit = fields.Char(
        string='有效限制',
        compute='_compute_effective_limit'
    )
    
    # 变更历史关联
    history_ids = fields.One2many(
        'apollo.package.quota.history',
        'package_quota_id',
        string='变更历史',
        readonly=True,
        help='额度的变更历史记录'
    )

    _sql_constraints = [
        ('package_quota_unique', 'unique(package_id, quota_definition_id)', 
         '同一套餐中的额度定义不能重复！'),
        ('quota_limit_positive', 'check(quota_limit >= 0)', 
         '额度限制数量不能为负数！'),
    ]

    @api.depends('package_id.name', 'quota_definition_id.quota_name')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            if record.package_id and record.quota_definition_id:
                record.display_name = f"{record.package_id.name} - {record.quota_definition_id.quota_name}"
            else:
                record.display_name = "新额度"

    @api.depends('quota_limit', 'is_unlimited', 'is_active', 'is_deleted')
    def _compute_effective_limit(self):
        """计算有效限制"""
        for record in self:
            if record.is_deleted:
                record.effective_limit = "已删除"
            elif not record.is_active:
                record.effective_limit = "已禁用"
            elif record.is_unlimited:
                record.effective_limit = "无限制"
            else:
                unit = record.quota_definition_id.unit or 'count'
                if unit == 'count':
                    record.effective_limit = str(record.quota_limit)
                else:
                    record.effective_limit = f"{record.quota_limit} {unit}"

    @api.constrains('quota_limit', 'is_unlimited')
    def _check_quota_limit(self):
        """检查额度限制"""
        for record in self:
            if not record.is_unlimited and record.quota_limit <= 0:
                raise ValidationError('非无限制额度的限制数量必须大于0！')

    @api.onchange('is_unlimited')
    def _onchange_is_unlimited(self):
        """当选择无限制时，自动设置额度为0"""
        if self.is_unlimited:
            self.quota_limit = 0

    @api.onchange('quota_definition_id')
    def _onchange_quota_definition_id(self):
        """当选择额度定义时，根据类型设置默认值"""
        if self.quota_definition_id:
            if self.quota_definition_id.quota_category == 'features':
                # 功能类额度默认为1（有权限）
                self.quota_limit = 1
            else:
                # 其他类型额度默认为0
                self.quota_limit = 0

    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = record.quota_definition_id.quota_name if record.quota_definition_id else "新额度"
            if record.effective_limit:
                name += f" ({record.effective_limit})"
            if record.is_additional:
                name += " [追加]"
            if not record.is_active:
                name += " [已禁用]"
            result.append((record.id, name))
        return result

    def toggle_active(self):
        """切换激活状态"""
        for record in self:
            record.is_active = not record.is_active

    @api.model_create_multi
    def create(self, vals_list):
        """重写创建方法，限制只能通过批量添加向导创建"""
        # 检查是否来自批量添加向导
        if not self.env.context.get('from_batch_add_wizard', False):
            raise UserError('额度定义只能通过"批量添加额度"功能进行添加，不允许直接创建套餐额度记录。')
        
        return super().create(vals_list)

    def write(self, vals):
        """重写写入方法，限制字段的修改"""
        # 检查是否尝试修改受保护的字段
        protected_fields = ['quota_definition_id', 'package_id']
        editable_fields = ['quota_limit', 'is_unlimited', 'is_additional', 'is_active']
        
        # 检查核心字段（额度定义和套餐）
        for field in protected_fields:
            if field in vals and not self.env.context.get('from_batch_add_wizard', False):
                raise UserError(f'不允许直接修改{self._fields[field].string}，请通过批量添加额度功能进行管理。')
        
        # 检查可编辑字段，只能通过编辑向导修改
        for field in editable_fields:
            if field in vals and not (
                self.env.context.get('from_batch_add_wizard', False) or 
                self.env.context.get('from_edit_wizard', False) or
                self.env.context.get('from_soft_delete', False) or
                self.env.context.get('from_restore', False)
            ):
                raise UserError(f'不允许直接修改{self._fields[field].string}，请通过"编辑"按钮进行修改。')
        
        return super().write(vals)

    def action_edit_quota(self):
        """打开编辑额度向导"""
        return {
            'type': 'ir.actions.act_window',
            'name': '编辑额度',
            'res_model': 'apollo.edit.quota.wizard',
            'view_mode': 'form',
            'target': 'new',
            'context': {'active_id': self.id}
        }

    def action_remove_quota(self):
        """移除额度（软删除）"""
        for record in self:
            if not record.is_deleted:
                # 记录删除前的状态
                old_values = {
                    'quota_limit': record.quota_limit,
                    'is_unlimited': record.is_unlimited,
                    'is_additional': record.is_additional,
                    'is_active': record.is_active,
                }
                
                new_values = {
                    'quota_limit': record.quota_limit,
                    'is_unlimited': record.is_unlimited,
                    'is_additional': record.is_additional,
                    'is_active': False,  # 删除时设为不活跃
                }
                
                # 创建删除历史记录
                history_model = self.env['apollo.package.quota.history']
                history_model.create({
                    'package_quota_id': record.id,
                    'old_quota_limit': old_values['quota_limit'],
                    'old_is_unlimited': old_values['is_unlimited'],
                    'old_is_additional': old_values['is_additional'],
                    'old_is_active': old_values['is_active'],
                    'new_quota_limit': new_values['quota_limit'],
                    'new_is_unlimited': new_values['is_unlimited'],
                    'new_is_additional': new_values['is_additional'],
                    'new_is_active': new_values['is_active'],
                    'change_reason': f'额度被移除：{record.quota_definition_id.quota_name}',
                    'change_type': 'delete',
                })
                
                # 执行软删除
                record.with_context(from_soft_delete=True).write({
                    'is_deleted': True,
                    'is_active': False,
                })
        
        # 返回成功结果并自动刷新
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': '额度移除成功',
                'message': '已成功移除额度，可通过"已删除"过滤器查看并恢复。',
                'type': 'success',
                'sticky': False,
                'next': {'type': 'ir.actions.client', 'tag': 'reload'}
            }
        }

    def unlink(self):
        """禁止直接删除，引导用户使用移除功能"""
        # 检查是否是管理员进行真正的物理删除（比如卸载模块时）
        if self.env.context.get('force_delete', False):
            return super().unlink()
        
        raise UserError('不允许直接删除额度记录。请使用"移除"按钮来移除额度，这样可以保留变更历史并支持恢复。')

    def action_restore_quota(self):
        """恢复被删除的额度"""
        for record in self:
            if record.is_deleted:
                # 记录恢复操作
                history_model = self.env['apollo.package.quota.history']
                history_model.create({
                    'package_quota_id': record.id,
                    'old_quota_limit': record.quota_limit,
                    'old_is_unlimited': record.is_unlimited,
                    'old_is_additional': record.is_additional,
                    'old_is_active': False,
                    'new_quota_limit': record.quota_limit,
                    'new_is_unlimited': record.is_unlimited,
                    'new_is_additional': record.is_additional,
                    'new_is_active': True,
                    'change_reason': f'额度被恢复：{record.quota_definition_id.quota_name}',
                    'change_type': 'restore',
                })
                
                # 恢复额度
                record.with_context(from_restore=True).write({
                    'is_deleted': False,
                    'is_active': True,
                })
        
        return {
            'type': 'ir.actions.client',
            'tag': 'reload',
        }
