# -*- coding: utf-8 -*-

from odoo import models, fields, api


class PackageQuotaHistory(models.Model):
    """套餐额度变更历史记录"""
    _name = 'apollo.package.quota.history'
    _description = '套餐额度变更历史'
    _order = 'create_date desc'
    _rec_name = 'display_name'

    package_quota_id = fields.Many2one(
        'apollo.package.quota',
        string='套餐额度',
        required=True,
        ondelete='cascade',
        help='关联的套餐额度记录'
    )
    
    # 变更前的值
    old_quota_limit = fields.Integer(
        string='变更前额度限制',
        help='变更前的额度限制数量'
    )
    old_is_unlimited = fields.Boolean(
        string='变更前是否无限制',
        help='变更前的无限制状态'
    )
    old_is_additional = fields.Boolean(
        string='变更前是否追加额度',
        help='变更前的追加额度状态'
    )
    old_is_active = fields.Boolean(
        string='变更前是否启用',
        help='变更前的启用状态'
    )
    
    # 变更后的值
    new_quota_limit = fields.Integer(
        string='变更后额度限制',
        help='变更后的额度限制数量'
    )
    new_is_unlimited = fields.Boolean(
        string='变更后是否无限制',
        help='变更后的无限制状态'
    )
    new_is_additional = fields.Boolean(
        string='变更后是否追加额度',
        help='变更后的追加额度状态'
    )
    new_is_active = fields.Boolean(
        string='变更后是否启用',
        help='变更后的启用状态'
    )
    
    change_reason = fields.Text(
        string='变更原因',
        help='变更的原因说明'
    )
    
    change_type = fields.Selection([
        ('quota_limit', '额度限制变更'),
        ('unlimited', '无限制状态变更'),
        ('additional', '追加额度状态变更'),
        ('active', '启用状态变更'),
        ('multiple', '多项变更'),
        ('delete', '额度删除'),
        ('restore', '额度恢复'),
    ], string='变更类型', required=True, help='变更的类型')
    
    changed_by = fields.Many2one(
        'res.users',
        string='变更人',
        required=True,
        default=lambda self: self.env.user,
        help='执行变更的用户'
    )
    
    change_date = fields.Datetime(
        string='变更时间',
        required=True,
        default=fields.Datetime.now,
        help='变更发生的时间'
    )
    
    # 关联字段 - 存储重要的外键关联
    package_id = fields.Many2one(
        related='package_quota_id.package_id',
        string='套餐',
        readonly=True,
        store=True
    )
    
    quota_definition_id = fields.Many2one(
        related='package_quota_id.quota_definition_id',
        string='额度定义',
        readonly=True,
        store=True
    )
    
    display_name = fields.Char(
        string='显示名称',
        compute='_compute_display_name',
        store=True
    )
    
    @api.depends('package_id.name', 'quota_definition_id.quota_name', 'change_type', 'change_date')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            change_type_dict = dict(record._fields['change_type'].selection)
            change_type_name = change_type_dict.get(record.change_type, '未知变更')
            package_name = record.package_id.name if record.package_id else '未知套餐'
            quota_name = record.quota_definition_id.quota_name if record.quota_definition_id else '未知额度'
            record.display_name = f"{package_name} - {quota_name} - {change_type_name}"
    
    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = record.display_name or "额度变更记录"
            result.append((record.id, name))
        return result
    
    @api.model
    def create_history_record(self, package_quota, old_values, new_values, reason=""):
        """创建历史记录"""
        # 确定变更类型
        changed_fields = []
        if old_values.get('quota_limit') != new_values.get('quota_limit'):
            changed_fields.append('quota_limit')
        if old_values.get('is_unlimited') != new_values.get('is_unlimited'):
            changed_fields.append('unlimited')
        if old_values.get('is_additional') != new_values.get('is_additional'):
            changed_fields.append('additional')
        if old_values.get('is_active') != new_values.get('is_active'):
            changed_fields.append('active')
        
        if len(changed_fields) == 1:
            change_type = changed_fields[0]
        elif len(changed_fields) > 1:
            change_type = 'multiple'
        else:
            return False  # 没有实际变更
        
        # 创建历史记录
        return self.create({
            'package_quota_id': package_quota.id,
            'old_quota_limit': old_values.get('quota_limit', 0),
            'old_is_unlimited': old_values.get('is_unlimited', False),
            'old_is_additional': old_values.get('is_additional', False),
            'old_is_active': old_values.get('is_active', True),
            'new_quota_limit': new_values.get('quota_limit', 0),
            'new_is_unlimited': new_values.get('is_unlimited', False),
            'new_is_additional': new_values.get('is_additional', False),
            'new_is_active': new_values.get('is_active', True),
            'change_reason': reason,
            'change_type': change_type,
        })
