# -*- coding: utf-8 -*-

from odoo import models, fields, api


class QuotaDefinition(models.Model):
    """额度定义表 - 定义所有可能的额度类型"""
    _name = 'apollo.quota.definition'
    _description = '额度定义'
    _order = 'quota_category, sequence, id'
    _rec_name = 'quota_name'

    quota_code = fields.Char(
        string='额度编码',
        required=True,
        size=50,
        help='额度的唯一标识符'
    )
    quota_name = fields.Char(
        string='额度名称',
        required=True,
        size=100,
        help='额度的显示名称'
    )
    quota_category = fields.Selection([
        ('accounts', '账户类'),
        ('conversations', '对话类'),
        ('actions', '操作类'),
        ('features', '功能类'),
    ], string='额度分类', required=True, help='额度的分类')
    
    feature_module_id = fields.Many2one(
        'apollo.feature.module',
        string='关联功能模块',
        help='关联的功能模块，用于功能权限控制'
    )
    description = fields.Text(
        string='额度描述',
        help='额度的详细描述'
    )
    unit = fields.Char(
        string='计量单位',
        size=20,
        default='count',
        help='额度的计量单位，如：count, hours, days等'
    )
    is_active = fields.Boolean(
        string='是否激活',
        default=True,
        help='是否激活此额度定义'
    )
    sequence = fields.Integer(
        string='排序序号',
        default=10,
        help='用于排序的序号'
    )
    created_date = fields.Datetime(
        string='创建时间',
        default=fields.Datetime.now,
        readonly=True
    )

    # 关联字段
    package_quota_ids = fields.One2many(
        'apollo.package.quota',
        'quota_definition_id',
        string='套餐额度'
    )
    quota_adjustment_ids = fields.One2many(
        'apollo.organization.quota.adjustment',
        'quota_definition_id',
        string='额度调整记录'
    )
    quota_usage_ids = fields.One2many(
        'apollo.organization.quota.usage',
        'quota_definition_id',
        string='额度使用记录'
    )

    _sql_constraints = [
        ('quota_code_unique', 'unique(quota_code)', '额度编码必须唯一！'),
    ]

    @api.model
    def name_search(self, name='', args=None, operator='ilike', limit=100):
        """支持按编码和名称搜索"""
        if args is None:
            args = []
        domain = args[:]
        if name:
            domain = ['|', ('quota_code', operator, name), ('quota_name', operator, name)] + domain
        records = self.search(domain, limit=limit, order=self._order)
        return records.name_get()

    def name_get(self):
        """自定义显示名称"""
        result = []
        for record in self:
            name = f"[{record.quota_code}] {record.quota_name}"
            if record.unit and record.unit != 'count':
                name += f" ({record.unit})"
            result.append((record.id, name))
        return result

    @api.depends('quota_category')
    def _compute_display_name(self):
        """计算显示名称"""
        for record in self:
            category_dict = dict(self._fields['quota_category'].selection)
            category_name = category_dict.get(record.quota_category, '')
            record.display_name = f"[{category_name}] {record.quota_name}"
