# -*- coding: utf-8 -*-

from odoo import models, api
from odoo.exceptions import AccessDenied, ValidationError


class PackagePermissionService(models.AbstractModel):
    """套餐权限服务 - 提供基于额度的权限检查功能"""
    _name = 'apollo.package.permission.service'
    _description = '套餐权限服务'

    @api.model
    def check_feature_access(self, user_id, module_code):
        """
        检查用户是否有权限使用某功能模块（通过组织）
        
        Args:
            user_id: 用户ID
            module_code: 功能模块编码
            
        Returns:
            dict: {'access': bool, 'message': str, 'organizations': list}
        """
        # 1. 获取用户所属的组织
        user_organizations = self._get_user_organizations(user_id)
        
        if not user_organizations:
            return {
                'access': False,
                'message': '用户不属于任何组织',
                'organizations': []
            }
        
        # 2. 获取功能模块相关的额度定义
        feature_quotas = self._get_feature_quotas(module_code)
        
        if not feature_quotas:
            return {
                'access': False,
                'message': f'功能模块 {module_code} 没有配置相关额度',
                'organizations': []
            }
        
        # 3. 检查任一组织是否有该功能权限
        accessible_orgs = []
        quota_service = self.env['apollo.package.quota.service']
        
        for org in user_organizations:
            org_quotas = quota_service.get_organization_effective_quotas(org.id)
            
            # 检查组织是否拥有任一相关额度且额度值大于0
            for quota in feature_quotas:
                quota_code = quota.quota_code
                if quota_code in org_quotas:
                    quota_info = org_quotas[quota_code]
                    if quota_info['is_unlimited'] or quota_info['limit'] > 0:
                        accessible_orgs.append({
                            'id': org.id,
                            'name': org.name,
                            'quota_code': quota_code,
                            'quota_name': quota.quota_name
                        })
                        break
        
        return {
            'access': len(accessible_orgs) > 0,
            'message': f'在 {len(accessible_orgs)} 个组织中有访问权限' if accessible_orgs else '没有访问权限',
            'organizations': accessible_orgs
        }

    @api.model
    def check_quota_permission(self, user_id, quota_code, required_amount=1, organization_id=None):
        """
        检查用户是否有足够的额度使用某功能
        
        Args:
            user_id: 用户ID
            quota_code: 额度编码
            required_amount: 需要的额度数量
            organization_id: 指定组织ID，如果为空则检查所有组织
            
        Returns:
            dict: {'access': bool, 'message': str, 'quota_info': dict, 'organization': dict}
        """
        # 获取用户所属的组织
        user_organizations = self._get_user_organizations(user_id)
        
        if not user_organizations:
            return {
                'access': False,
                'message': '用户不属于任何组织',
                'quota_info': {},
                'organization': None
            }
        
        # 如果指定了组织，检查用户是否属于该组织
        if organization_id:
            target_orgs = user_organizations.filtered(lambda o: o.id == organization_id)
            if not target_orgs:
                return {
                    'access': False,
                    'message': '用户不属于指定组织',
                    'quota_info': {},
                    'organization': None
                }
            check_orgs = target_orgs
        else:
            check_orgs = user_organizations
        
        quota_service = self.env['apollo.package.quota.service']
        
        # 检查各个组织的额度
        for org in check_orgs:
            quota_check = quota_service.check_quota_available(
                org.id, quota_code, required_amount
            )
            
            if quota_check['available']:
                return {
                    'access': True,
                    'message': f'组织 {org.name} 有足够额度',
                    'quota_info': quota_check,
                    'organization': {'id': org.id, 'name': org.name}
                }
        
        # 如果没有组织有足够额度，返回第一个组织的额度信息
        first_org = check_orgs[0]
        quota_check = quota_service.check_quota_available(
            first_org.id, quota_code, required_amount
        )
        
        return {
            'access': False,
            'message': f'额度不足：{quota_check["message"]}',
            'quota_info': quota_check,
            'organization': {'id': first_org.id, 'name': first_org.name}
        }

    @api.model
    def consume_user_quota(self, user_id, quota_code, amount=1, organization_id=None):
        """
        消费用户额度
        
        Args:
            user_id: 用户ID
            quota_code: 额度编码
            amount: 消费数量
            organization_id: 指定组织ID，如果为空则使用第一个有权限的组织
            
        Returns:
            dict: {'success': bool, 'message': str, 'organization': dict}
        """
        # 先检查权限
        permission_check = self.check_quota_permission(
            user_id, quota_code, amount, organization_id
        )
        
        if not permission_check['access']:
            return {
                'success': False,
                'message': permission_check['message'],
                'organization': permission_check['organization']
            }
        
        # 消费额度
        target_org = permission_check['organization']
        quota_service = self.env['apollo.package.quota.service']
        
        consume_result = quota_service.consume_quota(
            target_org['id'], quota_code, amount
        )
        
        return {
            'success': consume_result['success'],
            'message': consume_result['message'],
            'organization': target_org
        }

    @api.model
    def get_user_quota_summary(self, user_id, organization_id=None):
        """
        获取用户额度汇总
        
        Args:
            user_id: 用户ID
            organization_id: 指定组织ID，如果为空则返回所有组织的额度
            
        Returns:
            dict: 额度汇总信息
        """
        user_organizations = self._get_user_organizations(user_id)
        
        if not user_organizations:
            return {}
        
        if organization_id:
            target_orgs = user_organizations.filtered(lambda o: o.id == organization_id)
            if not target_orgs:
                return {}
            check_orgs = target_orgs
        else:
            check_orgs = user_organizations
        
        quota_service = self.env['apollo.package.quota.service']
        summary = {}
        
        for org in check_orgs:
            org_summary = quota_service.get_organization_quota_summary(org.id)
            summary[org.id] = {
                'organization_name': org.name,
                'quotas': org_summary
            }
        
        return summary

    @api.model
    def _get_user_organizations(self, user_id):
        """获取用户所属的组织"""
        return self.env['apollo.organization'].search([
            ('current_user_ids', 'in', [user_id])
        ])

    @api.model
    def _get_feature_quotas(self, module_code):
        """获取功能模块相关的额度定义"""
        return self.env['apollo.quota.definition'].search([
            ('feature_module_id.module_code', '=', module_code),
            ('is_active', '=', True)
        ])

    @api.model
    def require_feature_access(self, module_code, user_id=None):
        """
        装饰器辅助方法：要求功能访问权限
        
        Args:
            module_code: 功能模块编码
            user_id: 用户ID，默认为当前用户
            
        Raises:
            AccessDenied: 如果没有访问权限
        """
        if user_id is None:
            user_id = self.env.user.id
        
        access_check = self.check_feature_access(user_id, module_code)
        
        if not access_check['access']:
            raise AccessDenied(f'没有访问功能模块 {module_code} 的权限：{access_check["message"]}')

    @api.model
    def require_quota_available(self, quota_code, required_amount=1, user_id=None, organization_id=None):
        """
        装饰器辅助方法：要求额度可用
        
        Args:
            quota_code: 额度编码
            required_amount: 需要的额度数量
            user_id: 用户ID，默认为当前用户
            organization_id: 指定组织ID
            
        Raises:
            ValidationError: 如果额度不足
        """
        if user_id is None:
            user_id = self.env.user.id
        
        quota_check = self.check_quota_permission(
            user_id, quota_code, required_amount, organization_id
        )
        
        if not quota_check['access']:
            raise ValidationError(f'额度不足：{quota_check["message"]}')

    @api.model
    def get_organization_permissions(self, organization_id):
        """
        获取组织的所有权限信息
        
        Args:
            organization_id: 组织ID
            
        Returns:
            dict: 权限信息
        """
        quota_service = self.env['apollo.package.quota.service']
        effective_quotas = quota_service.get_organization_effective_quotas(organization_id)
        
        # 按分类组织权限信息
        permissions = {
            'accounts': {},
            'conversations': {},
            'actions': {},
            'features': {},
        }
        
        # 获取所有额度定义
        all_quotas = self.env['apollo.quota.definition'].search([('is_active', '=', True)])
        quota_dict = {q.quota_code: q for q in all_quotas}
        
        for quota_code, quota_info in effective_quotas.items():
            quota_def = quota_dict.get(quota_code)
            if quota_def:
                category = quota_def.quota_category
                permissions[category][quota_code] = {
                    'quota_name': quota_def.quota_name,
                    'quota_limit': quota_info['limit'],
                    'is_unlimited': quota_info['is_unlimited'],
                    'has_access': quota_info['is_unlimited'] or quota_info['limit'] > 0,
                    'unit': quota_def.unit,
                    'source': quota_info.get('source', 'unknown')
                }
        
        return permissions

    @api.model
    def check_bulk_permissions(self, user_id, permission_list):
        """
        批量检查权限
        
        Args:
            user_id: 用户ID
            permission_list: 权限列表 [{'type': 'feature'|'quota', 'code': str, 'amount': int}]
            
        Returns:
            dict: 批量检查结果
        """
        results = {}
        
        for perm in permission_list:
            perm_type = perm.get('type')
            code = perm.get('code')
            amount = perm.get('amount', 1)
            
            if perm_type == 'feature':
                result = self.check_feature_access(user_id, code)
            elif perm_type == 'quota':
                result = self.check_quota_permission(user_id, code, amount)
            else:
                result = {'access': False, 'message': f'未知权限类型：{perm_type}'}
            
            results[code] = result
        
        return results
