# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import UserError


class BatchAddQuotaWizard(models.TransientModel):
    """批量添加额度向导"""
    _name = 'apollo.batch.add.quota.wizard'
    _description = '批量添加额度向导'

    package_id = fields.Many2one(
        'apollo.package',
        string='目标套餐',
        required=True,
        readonly=True
    )
    
    quota_line_ids = fields.One2many(
        'apollo.batch.add.quota.line',
        'wizard_id',
        string='额度选择'
    )
    
    @api.model
    def default_get(self, fields_list):
        """设置默认值"""
        res = super().default_get(fields_list)
        package_id = self.env.context.get('active_id')
        if package_id:
            res['package_id'] = package_id
            # 获取所有可用的额度定义
            available_quotas = self._get_available_quotas(package_id)
            quota_lines = []
            for quota in available_quotas:
                quota_lines.append((0, 0, {
                    'quota_definition_id': quota.id,
                    'selected': False,
                    'quota_limit': self._get_default_limit(quota),
                    'is_unlimited': False,
                }))
            res['quota_line_ids'] = quota_lines
        return res
    
    def _get_available_quotas(self, package_id):
        """获取可用的额度定义（排除已添加且未删除的）"""
        package = self.env['apollo.package'].browse(package_id)
        # 只排除未删除的额度，允许已删除的额度重新出现在列表中
        existing_quota_ids = package.package_quota_ids.mapped('quota_definition_id.id')
        
        available_quotas = self.env['apollo.quota.definition'].search([
            ('is_active', '=', True),
            ('id', 'not in', existing_quota_ids)
        ])
        return available_quotas
    
    def _get_default_limit(self, quota):
        """根据额度类型获取默认限制"""
        if quota.quota_category == 'features':
            return 1  # 功能类额度默认为1
        elif quota.quota_category == 'accounts':
            return 10  # 账户类额度默认为10
        elif quota.quota_category == 'conversations':
            return 1000  # 对话类额度默认为1000
        elif quota.quota_category == 'actions':
            return 100  # 操作类额度默认为100
        else:
            return 1
    
    def action_add_quotas(self):
        """执行批量添加额度"""
        if not self.quota_line_ids.filtered('selected'):
            raise UserError('请至少选择一个额度定义！')
        
        package_quota_model = self.env['apollo.package.quota']
        added_count = 0
        
        errors = []
        restored_count = 0
        for line in self.quota_line_ids.filtered('selected'):
            try:
                # 首先检查是否已经存在已删除的额度记录
                existing_deleted = package_quota_model.search([
                    ('package_id', '=', self.package_id.id),
                    ('quota_definition_id', '=', line.quota_definition_id.id),
                    ('is_deleted', '=', True)
                ], limit=1)
                
                if existing_deleted:
                    # 记录恢复前的状态
                    old_quota_limit = existing_deleted.quota_limit
                    old_is_unlimited = existing_deleted.is_unlimited
                    old_is_additional = existing_deleted.is_additional
                    
                    # 恢复已删除的额度记录
                    existing_deleted.with_context(from_restore=True).write({
                        'quota_limit': line.quota_limit,
                        'is_unlimited': line.is_unlimited,
                        'is_active': True,
                        'is_deleted': False,
                    })
                    
                    # 记录恢复历史
                    history_model = self.env['apollo.package.quota.history']
                    history_model.create({
                        'package_quota_id': existing_deleted.id,
                        'old_quota_limit': old_quota_limit,
                        'old_is_unlimited': old_is_unlimited,
                        'old_is_additional': old_is_additional,
                        'old_is_active': False,
                        'new_quota_limit': line.quota_limit,
                        'new_is_unlimited': line.is_unlimited,
                        'new_is_additional': old_is_additional,
                        'new_is_active': True,
                        'change_reason': f'通过批量添加恢复额度：{line.quota_name}',
                        'change_type': 'restore',
                    })
                    restored_count += 1
                else:
                    # 创建新的额度记录
                    package_quota_model.with_context(from_batch_add_wizard=True).create({
                        'package_id': self.package_id.id,
                        'quota_definition_id': line.quota_definition_id.id,
                        'quota_limit': line.quota_limit,
                        'is_unlimited': line.is_unlimited,
                        'is_active': True,
                    })
                    added_count += 1
            except Exception as e:
                # 记录错误信息，继续处理其他额度
                errors.append(f"{line.quota_name}: {str(e)}")
                continue
        
        # 构建结果消息
        total_success = added_count + restored_count
        if total_success > 0:
            title = '批量添加完成'
            message_parts = []
            if added_count > 0:
                message_parts.append(f'新增 {added_count} 个额度')
            if restored_count > 0:
                message_parts.append(f'恢复 {restored_count} 个额度')
            message = '成功：' + '、'.join(message_parts)
            if errors:
                message += f'，{len(errors)} 个额度处理失败'
            notification_type = 'success'
        else:
            title = '批量添加失败'
            message = '没有成功添加或恢复任何额度'
            notification_type = 'warning'
        
        # 使用更安全的方式返回结果
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': title,
                'message': message,
                'type': notification_type,
                'sticky': False,
                'next': {'type': 'ir.actions.act_window_close'}
            }
        }


class BatchAddQuotaLine(models.TransientModel):
    """批量添加额度行"""
    _name = 'apollo.batch.add.quota.line'
    _description = '批量添加额度行'

    wizard_id = fields.Many2one(
        'apollo.batch.add.quota.wizard',
        string='向导',
        required=True,
        ondelete='cascade'
    )
    
    quota_definition_id = fields.Many2one(
        'apollo.quota.definition',
        string='额度定义',
        required=True
    )
    
    selected = fields.Boolean(
        string='选择',
        default=False
    )
    
    quota_limit = fields.Integer(
        string='额度限制',
        default=1
    )
    
    is_unlimited = fields.Boolean(
        string='无限制',
        default=False
    )
    

    
    @api.onchange('is_unlimited')
    def _onchange_is_unlimited(self):
        """当选择无限制时，设置额度为0"""
        if self.is_unlimited:
            self.quota_limit = 0
    
    @api.onchange('selected')
    def _onchange_selected(self):
        """当选择时，确保有合理的默认值"""
        if self.selected and not self.is_unlimited and self.quota_limit <= 0:
            if self.quota_definition_id:
                self.quota_limit = self.wizard_id._get_default_limit(self.quota_definition_id)
