# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import UserError


class EditQuotaWizard(models.TransientModel):
    """编辑额度向导"""
    _name = 'apollo.edit.quota.wizard'
    _description = '编辑额度向导'

    package_quota_id = fields.Many2one(
        'apollo.package.quota',
        string='套餐额度',
        required=True,
        readonly=True
    )
    
    # 当前值（只读，用于显示对比）
    current_quota_limit = fields.Integer(
        string='当前额度限制',
        readonly=True
    )
    current_is_unlimited = fields.Boolean(
        string='当前是否无限制',
        readonly=True
    )
    current_is_additional = fields.Boolean(
        string='当前是否追加额度',
        readonly=True
    )
    current_is_active = fields.Boolean(
        string='当前是否启用',
        readonly=True
    )
    
    # 新值（可编辑）
    new_quota_limit = fields.Integer(
        string='新额度限制',
        default=0
    )
    new_is_unlimited = fields.Boolean(
        string='新是否无限制',
        default=False
    )
    new_is_additional = fields.Boolean(
        string='新是否追加额度',
        default=False
    )
    new_is_active = fields.Boolean(
        string='新是否启用',
        default=True
    )
    
    change_reason = fields.Text(
        string='变更原因',
        help='请说明变更的原因',
        required=True
    )
    

    
    @api.model
    def default_get(self, fields_list):
        """设置默认值"""
        res = super().default_get(fields_list)
        package_quota_id = self.env.context.get('active_id')
        if package_quota_id:
            package_quota = self.env['apollo.package.quota'].browse(package_quota_id)
            res.update({
                'package_quota_id': package_quota.id,
                'current_quota_limit': package_quota.quota_limit,
                'current_is_unlimited': package_quota.is_unlimited,
                'current_is_additional': package_quota.is_additional,
                'current_is_active': package_quota.is_active,
                'new_quota_limit': package_quota.quota_limit,
                'new_is_unlimited': package_quota.is_unlimited,
                'new_is_additional': package_quota.is_additional,
                'new_is_active': package_quota.is_active,
            })
        return res
    
    @api.onchange('new_is_unlimited')
    def _onchange_new_is_unlimited(self):
        """当选择无限制时，自动设置额度为0"""
        if self.new_is_unlimited:
            self.new_quota_limit = 0
    
    def action_apply_changes(self):
        """应用变更"""
        if not self.package_quota_id:
            raise UserError('未找到要编辑的额度记录！')
        
        # 检查是否有实际变更
        old_values = {
            'quota_limit': self.current_quota_limit,
            'is_unlimited': self.current_is_unlimited,
            'is_additional': self.current_is_additional,
            'is_active': self.current_is_active,
        }
        
        new_values = {
            'quota_limit': self.new_quota_limit,
            'is_unlimited': self.new_is_unlimited,
            'is_additional': self.new_is_additional,
            'is_active': self.new_is_active,
        }
        
        # 检查是否有变更
        has_changes = False
        for key in old_values:
            if old_values[key] != new_values[key]:
                has_changes = True
                break
        
        if not has_changes:
            raise UserError('没有检测到任何变更！')
        
        # 验证新值
        if not self.new_is_unlimited and self.new_quota_limit <= 0:
                            raise UserError('非无限制额度的限制数量必须大于0！')
        
        # 创建变更历史记录
        history_model = self.env['apollo.package.quota.history']
        history_model.create_history_record(
            self.package_quota_id,
            old_values,
            new_values,
            self.change_reason
        )
        
        # 应用变更（使用特殊上下文绕过写入限制）
        self.package_quota_id.with_context(from_edit_wizard=True).write({
            'quota_limit': self.new_quota_limit,
            'is_unlimited': self.new_is_unlimited,
            'is_additional': self.new_is_additional,
            'is_active': self.new_is_active,
        })
        
        # 显示成功通知并关闭向导
        change_summary = self._get_change_summary(old_values, new_values)
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': '额度修改成功',
                'message': f'已成功修改 {self.quota_name} 的额度设置：{change_summary}',
                'type': 'success',
                'sticky': False,
                'next': {'type': 'ir.actions.act_window_close'}
            }
        }
    
    def _get_change_summary(self, old_values, new_values):
        """获取变更摘要"""
        changes = []
        
        if old_values['quota_limit'] != new_values['quota_limit']:
            changes.append(f"额度限制：{old_values['quota_limit']} → {new_values['quota_limit']}")
        
        if old_values['is_unlimited'] != new_values['is_unlimited']:
            old_unlimited = "是" if old_values['is_unlimited'] else "否"
            new_unlimited = "是" if new_values['is_unlimited'] else "否"
            changes.append(f"无限制：{old_unlimited} → {new_unlimited}")
        
        if old_values['is_additional'] != new_values['is_additional']:
            old_additional = "是" if old_values['is_additional'] else "否"
            new_additional = "是" if new_values['is_additional'] else "否"
            changes.append(f"追加额度：{old_additional} → {new_additional}")
        
        if old_values['is_active'] != new_values['is_active']:
            old_active = "是" if old_values['is_active'] else "否"
            new_active = "是" if new_values['is_active'] else "否"
            changes.append(f"启用状态：{old_active} → {new_active}")
        
        return "; ".join(changes)
