# -*- coding: utf-8 -*-

from odoo import models, fields, api, _
from odoo.exceptions import ValidationError
from datetime import timedelta


class OrganizationPackageConfigWizard(models.TransientModel):
    """组织套餐配置向导"""
    _name = 'apollo.organization.package.config.wizard'
    _description = '组织套餐配置向导'

    organization_id = fields.Many2one(
        'apollo.organization',
        string='组织',
        required=True,
        readonly=True
    )
    action_type = fields.Selection([
        ('new_subscription', '新建订阅'),
        ('renew_subscription', '续费订阅'),
        ('change_package', '更换套餐'),
        ('extend_subscription', '延长订阅'),
    ], string='操作类型', required=True, default='new_subscription')
    
    # 套餐选择
    package_id = fields.Many2one(
        'apollo.package',
        string='选择套餐',
        required=True,
        domain="[('is_active', '=', True)]"
    )
    
    # 订阅日期
    start_date = fields.Date(
        string='开始日期',
        required=True,
        default=fields.Date.today
    )
    end_date = fields.Date(
        string='结束日期',
        compute='_compute_end_date',
        store=True,
        readonly=False
    )
    
    # 延长天数（仅用于延长订阅）
    extension_days = fields.Integer(
        string='延长天数',
        default=30,
        help='延长订阅的天数'
    )
    
    # 当前订阅信息
    current_subscription_id = fields.Many2one(
        'apollo.organization.package.subscription',
        string='当前订阅',
        compute='_compute_current_subscription'
    )
    current_package_id = fields.Many2one(
        'apollo.package',
        string='当前套餐',
        compute='_compute_current_subscription'
    )
    current_end_date = fields.Date(
        string='当前结束日期',
        compute='_compute_current_subscription'
    )
    
    # 操作说明
    reason = fields.Text(string='备注说明')
    
    # 信息字段
    info_message = fields.Html(
        string='操作说明',
        compute='_compute_info_message'
    )

    @api.depends('organization_id')
    def _compute_current_subscription(self):
        """计算当前订阅信息"""
        for record in self:
            if record.organization_id:
                active_subs = record.organization_id.active_package_subscription_ids.filtered('is_active_subscription')
                if active_subs:
                    latest_sub = active_subs.sorted('start_date', reverse=True)[0]
                    record.current_subscription_id = latest_sub
                    record.current_package_id = latest_sub.package_id
                    record.current_end_date = latest_sub.final_end_date
                else:
                    record.current_subscription_id = False
                    record.current_package_id = False
                    record.current_end_date = False
            else:
                record.current_subscription_id = False
                record.current_package_id = False
                record.current_end_date = False

    @api.depends('package_id', 'start_date')
    def _compute_end_date(self):
        """根据套餐时长计算结束日期"""
        for record in self:
            if record.package_id and record.start_date and record.package_id.duration_weeks:
                end_date = record.start_date + timedelta(weeks=record.package_id.duration_weeks)
                record.end_date = end_date
            elif record.action_type == 'extend_subscription' and record.current_end_date and record.extension_days:
                record.end_date = record.current_end_date + timedelta(days=record.extension_days)
            else:
                record.end_date = False

    @api.depends('action_type', 'organization_id', 'package_id', 'current_subscription_id')
    def _compute_info_message(self):
        """计算操作说明信息"""
        for record in self:
            if not record.organization_id:
                record.info_message = '<p>请选择组织</p>'
                continue
                
            messages = []
            
            if record.action_type == 'new_subscription':
                if record.current_subscription_id:
                    messages.append('<div class="alert alert-warning">注意：该组织已有活跃订阅，新建订阅将与现有订阅并存。</div>')
                messages.append('<p><strong>新建订阅说明：</strong></p>')
                messages.append('<ul>')
                messages.append('<li>为组织创建新的套餐订阅</li>')
                messages.append('<li>选择套餐后会自动计算订阅期限</li>')
                messages.append('<li>订阅创建后立即生效</li>')
                messages.append('</ul>')
                
            elif record.action_type == 'renew_subscription':
                if not record.current_subscription_id:
                    messages.append('<div class="alert alert-danger">错误：该组织没有活跃的订阅可以续费。</div>')
                else:
                    messages.append('<p><strong>续费订阅说明：</strong></p>')
                    messages.append('<ul>')
                    messages.append(f'<li>当前套餐：{record.current_package_id.name}</li>')
                    messages.append(f'<li>当前结束日期：{record.current_end_date}</li>')
                    messages.append('<li>续费将在当前订阅结束后开始</li>')
                    messages.append('</ul>')
                    
            elif record.action_type == 'change_package':
                if not record.current_subscription_id:
                    messages.append('<div class="alert alert-danger">错误：该组织没有活跃的订阅可以更换套餐。</div>')
                else:
                    messages.append('<p><strong>更换套餐说明：</strong></p>')
                    messages.append('<ul>')
                    messages.append(f'<li>当前套餐：{record.current_package_id.name}</li>')
                    messages.append('<li>更换套餐将取消当前订阅</li>')
                    messages.append('<li>新订阅将从指定日期开始</li>')
                    messages.append('</ul>')
                    
            elif record.action_type == 'extend_subscription':
                if not record.current_subscription_id:
                    messages.append('<div class="alert alert-danger">错误：该组织没有活跃的订阅可以延长。</div>')
                else:
                    messages.append('<p><strong>延长订阅说明：</strong></p>')
                    messages.append('<ul>')
                    messages.append(f'<li>当前套餐：{record.current_package_id.name}</li>')
                    messages.append(f'<li>当前结束日期：{record.current_end_date}</li>')
                    messages.append('<li>延长订阅将在当前结束日期基础上增加指定天数</li>')
                    messages.append('</ul>')
            
            record.info_message = ''.join(messages)

    @api.onchange('action_type')
    def _onchange_action_type(self):
        """根据操作类型调整默认值"""
        if self.action_type == 'renew_subscription' and self.current_package_id:
            self.package_id = self.current_package_id
            if self.current_end_date:
                self.start_date = self.current_end_date + timedelta(days=1)
        elif self.action_type == 'change_package':
            self.package_id = False
        elif self.action_type == 'extend_subscription':
            if self.current_package_id:
                self.package_id = self.current_package_id

    @api.onchange('extension_days')
    def _onchange_extension_days(self):
        """延长天数变化时重新计算结束日期"""
        if self.action_type == 'extend_subscription' and self.current_end_date and self.extension_days:
            self.end_date = self.current_end_date + timedelta(days=self.extension_days)

    def action_confirm(self):
        """确认配置"""
        self.ensure_one()
        
        if self.action_type == 'new_subscription':
            return self._create_new_subscription()
        elif self.action_type == 'renew_subscription':
            return self._renew_subscription()
        elif self.action_type == 'change_package':
            return self._change_package()
        elif self.action_type == 'extend_subscription':
            return self._extend_subscription()

    def _create_new_subscription(self):
        """创建新订阅"""
        if not self.package_id or not self.start_date or not self.end_date:
            raise ValidationError(_('请完整填写套餐信息'))
        
        subscription = self.env['apollo.organization.package.subscription'].create({
            'organization_id': self.organization_id.id,
            'package_id': self.package_id.id,
            'start_date': self.start_date,
            'end_date': self.end_date,
            'status': 'active',
        })
        
        return {
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.package.subscription',
            'res_id': subscription.id,
            'view_mode': 'form',
            'target': 'current',
        }

    def _renew_subscription(self):
        """续费订阅"""
        if not self.current_subscription_id:
            raise ValidationError(_('没有找到可续费的订阅'))
        
        if not self.package_id or not self.start_date or not self.end_date:
            raise ValidationError(_('请完整填写续费信息'))
        
        subscription = self.env['apollo.organization.package.subscription'].create({
            'organization_id': self.organization_id.id,
            'package_id': self.package_id.id,
            'start_date': self.start_date,
            'end_date': self.end_date,
            'status': 'active',
        })
        
        return {
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.package.subscription',
            'res_id': subscription.id,
            'view_mode': 'form',
            'target': 'current',
        }

    def _change_package(self):
        """更换套餐"""
        if not self.current_subscription_id:
            raise ValidationError(_('没有找到可更换的订阅'))
        
        if not self.package_id or not self.start_date or not self.end_date:
            raise ValidationError(_('请完整填写新套餐信息'))
        
        # 取消当前订阅
        self.current_subscription_id.action_cancel_subscription()
        
        # 创建新订阅
        subscription = self.env['apollo.organization.package.subscription'].create({
            'organization_id': self.organization_id.id,
            'package_id': self.package_id.id,
            'start_date': self.start_date,
            'end_date': self.end_date,
            'status': 'active',
        })
        
        return {
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.package.subscription',
            'res_id': subscription.id,
            'view_mode': 'form',
            'target': 'current',
        }

    def _extend_subscription(self):
        """延长订阅"""
        if not self.current_subscription_id:
            raise ValidationError(_('没有找到可延长的订阅'))
        
        if not self.extension_days or self.extension_days <= 0:
            raise ValidationError(_('延长天数必须大于0'))
        
        # 创建延长记录
        extension = self.env['apollo.organization.subscription.extension'].create({
            'subscription_id': self.current_subscription_id.id,
            'extension_days': self.extension_days,
            'original_end_date': self.current_subscription_id.final_end_date,
            'new_end_date': self.end_date,
            'reason': self.reason or '管理员延长订阅',
            'is_active': True,
        })
        
        return {
            'type': 'ir.actions.act_window',
            'res_model': 'apollo.organization.subscription.extension',
            'res_id': extension.id,
            'view_mode': 'form',
            'target': 'current',
        }
