# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import ValidationError, UserError


class PricingChangeWizard(models.TransientModel):
    """定价修改向导 - 用于修改套餐定价并记录变更"""
    _name = 'apollo.pricing.change.wizard'
    _description = '定价修改向导'

    # 基本信息
    pricing_id = fields.Many2one(
        'apollo.package.pricing',
        string='定价记录',
        required=True,
        readonly=True,
        help='要修改的定价记录'
    )
    package_id = fields.Many2one(
        'apollo.package',
        string='套餐',
        related='pricing_id.package_id',
        readonly=True
    )
    
    # 当前值显示
    current_price = fields.Float(
        string='当前价格',
        related='pricing_id.price',
        readonly=True
    )
    current_currency = fields.Selection(
        related='pricing_id.currency',
        string='当前货币',
        readonly=True
    )
    current_effective_date = fields.Date(
        string='当前生效日期',
        related='pricing_id.effective_date',
        readonly=True
    )
    current_expiry_date = fields.Date(
        string='当前失效日期',
        related='pricing_id.expiry_date',
        readonly=True
    )

    
    # 新值输入
    new_price = fields.Float(
        string='新价格',
        required=True,
        default=0.0,
        help='新的价格'
    )
    new_currency = fields.Selection([
        ('CNY', '人民币'),
        ('USD', '美元'),
        ('EUR', '欧元'),
    ], string='新货币类型', required=True, default='CNY', help='新的货币类型')
    
    new_effective_date = fields.Date(
        string='新生效日期',
        required=True,
        default=fields.Date.today,
        help='新的生效日期'
    )
    new_expiry_date = fields.Date(
        string='新失效日期',
        help='新的失效日期，为空表示永久有效'
    )

    
    # 变更信息
    change_reason = fields.Selection([
        ('market_adjustment', '市场调整'),
        ('cost_change', '成本变化'),
        ('promotion', '促销活动'),
        ('business_strategy', '商业策略'),
        ('error_correction', '错误修正'),
        ('customer_request', '客户要求'),
        ('policy_change', '政策变化'),
        ('other', '其他'),
    ], string='变更原因', required=True, help='定价变更原因')
    
    custom_reason = fields.Char(
        string='自定义原因',
        help='当选择"其他"时，请输入具体原因'
    )
    notes = fields.Text(
        string='备注',
        help='额外说明信息'
    )
    
    # 计算字段
    price_change_amount = fields.Float(
        string='价格变化金额',
        compute='_compute_price_changes'
    )
    price_change_percentage = fields.Float(
        string='价格变化百分比',
        compute='_compute_price_changes',
        digits=(16, 4),  # 设置精度：总共16位，小数点后4位
        help='价格变化的百分比，以小数形式显示（如0.2表示20%）'
    )
    has_changes = fields.Boolean(
        string='有变更',
        compute='_compute_has_changes'
    )
    
    @api.model
    def default_get(self, fields_list):
        """设置默认值"""
        defaults = super().default_get(fields_list)
        
        # 获取当前定价记录
        pricing_id = self.env.context.get('active_id')
        if pricing_id:
            pricing = self.env['apollo.package.pricing'].browse(pricing_id)
            if pricing.exists():
                defaults.update({
                    'pricing_id': pricing.id,
                    'new_price': pricing.price,
                    'new_currency': pricing.currency,
                    'new_effective_date': pricing.effective_date,
                    'new_expiry_date': pricing.expiry_date,
                })
        
        return defaults
    
    @api.depends('current_price', 'new_price')
    def _compute_price_changes(self):
        """计算价格变化"""
        for record in self:
            # 初始化默认值
            record.price_change_amount = 0.0
            record.price_change_percentage = 0.0
            
            # 检查有效的价格值
            if (record.current_price is not False and record.new_price is not False and 
                record.current_price is not None and record.new_price is not None):
                
                # 计算价格变化金额
                record.price_change_amount = record.new_price - record.current_price
                
                # 计算百分比变化 (Odoo的percentage widget期望0-1之间的值)
                if record.current_price > 0:
                    percentage = record.price_change_amount / record.current_price
                    # 限制精度，避免浮点数计算误差
                    record.price_change_percentage = round(percentage, 6)
                elif record.current_price == 0 and record.new_price > 0:
                    # 从0价格变为正价格，显示为100%(即1.0)
                    record.price_change_percentage = 1.0
                else:
                    record.price_change_percentage = 0.0
    
    @api.depends('current_price', 'new_price', 'current_currency', 'new_currency',
                 'current_effective_date', 'new_effective_date', 'current_expiry_date', 'new_expiry_date')
    def _compute_has_changes(self):
        """检查是否有变更"""
        for record in self:
            record.has_changes = (
                record.current_price != record.new_price or
                record.current_currency != record.new_currency or
                record.current_effective_date != record.new_effective_date or
                record.current_expiry_date != record.new_expiry_date
            )
    
    @api.constrains('new_price')
    def _check_new_price(self):
        """检查新价格"""
        for record in self:
            if record.new_price < 0:
                raise ValidationError('价格不能为负数！')
    
    @api.constrains('new_effective_date', 'new_expiry_date')
    def _check_new_date_range(self):
        """检查新日期范围"""
        for record in self:
            if record.new_expiry_date and record.new_effective_date > record.new_expiry_date:
                raise ValidationError('生效日期不能晚于失效日期！')
    
    @api.constrains('change_reason', 'custom_reason')
    def _check_custom_reason(self):
        """检查自定义原因"""
        for record in self:
            if record.change_reason == 'other' and not record.custom_reason:
                raise ValidationError('选择"其他"原因时，必须填写自定义原因！')
    
    def _check_overlapping_pricing(self):
        """检查是否与其他定价重叠"""
        for record in self:
            # 检查同一套餐的其他定价是否重叠
            domain = [
                ('package_id', '=', record.package_id.id),
                ('id', '!=', record.pricing_id.id),
                ('effective_date', '<=', record.new_expiry_date or '9999-12-31'),
            ]
            
            if record.new_expiry_date:
                domain.append(('expiry_date', '>=', record.new_effective_date))
            else:
                domain.append('|')
                domain.append(('expiry_date', '>=', record.new_effective_date))
                domain.append(('expiry_date', '=', False))
            
            overlapping = self.env['apollo.package.pricing'].search(domain)
            if overlapping:
                raise ValidationError(f'套餐 {record.package_id.name} 在此时间段内已有其他定价！')
    
    def action_confirm_change(self):
        """确认修改"""
        self.ensure_one()
        
        if not self.has_changes:
            raise UserError('没有检测到任何变更！')
        
        # 检查重叠
        self._check_overlapping_pricing()
        
        # 保存旧值
        old_values = {
            'price': self.current_price,
            'currency': self.current_currency,
            'effective_date': self.current_effective_date,
            'expiry_date': self.current_expiry_date,
        }
        
        # 更新定价记录
        update_vals = {
            'price': self.new_price,
            'currency': self.new_currency,
            'effective_date': self.new_effective_date,
            'expiry_date': self.new_expiry_date,
        }
        
        # 执行更新
        self.pricing_id.sudo().write(update_vals)
        
        # 创建变更记录
        reason = self.custom_reason if self.change_reason == 'other' else dict(self._fields['change_reason'].selection)[self.change_reason]
        
        self.env['apollo.package.pricing.change.log'].create_change_log(
            pricing_record=self.pricing_id,
            change_type='update',
            old_values=old_values,
            reason=reason,
            notes=self.notes
        )
        
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': '成功',
                'message': f'套餐 {self.package_id.name} 的定价已成功修改！',
                'type': 'success',
                'sticky': False,
                'next': {'type': 'ir.actions.act_window_close'}
            }
        }
    
    def action_cancel(self):
        """取消修改"""
        return {'type': 'ir.actions.act_window_close'}
