# -*- coding: utf-8 -*-

from odoo import models, fields, api
from odoo.exceptions import UserError, ValidationError


class QuickAddPricingWizard(models.TransientModel):
    """快速添加套餐定价向导"""
    _name = 'apollo.quick.add.pricing.wizard'
    _description = '快速添加套餐定价向导'

    package_id = fields.Many2one(
        'apollo.package',
        string='套餐',
        required=True,
        readonly=True
    )
    price = fields.Float(
        string='价格',
        required=True,
        default=0.0,
        help='套餐价格'
    )
    currency = fields.Selection([
        ('CNY', '人民币'),
        ('USD', '美元'),
        ('EUR', '欧元'),
    ], string='货币类型', required=True, default='CNY', help='价格货币类型')
    

    effective_date = fields.Date(
        string='生效日期',
        required=True,
        default=fields.Date.today,
        help='价格生效日期'
    )
    expiry_date = fields.Date(
        string='失效日期',
        help='价格失效日期，为空表示永久有效'
    )
    

    
    # 计算字段
    formatted_price = fields.Char(
        string='格式化价格',
        compute='_compute_formatted_price'
    )
    
    @api.depends('price', 'currency')
    def _compute_formatted_price(self):
        """计算格式化价格"""
        for record in self:
            currency_symbol = {'CNY': '¥', 'USD': '$', 'EUR': '€'}.get(record.currency, '')
            record.formatted_price = f"{currency_symbol}{record.price:,.2f}"
    
    @api.model
    def default_get(self, fields_list):
        """设置默认值"""
        res = super().default_get(fields_list)
        package_id = self.env.context.get('active_id')
        if package_id:
            res['package_id'] = package_id
        return res
    
    @api.constrains('effective_date', 'expiry_date')
    def _check_date_range(self):
        """检查日期范围"""
        for record in self:
            if record.expiry_date and record.effective_date > record.expiry_date:
                raise ValidationError('生效日期不能晚于失效日期！')
    
    @api.constrains('price')
    def _check_price(self):
        """检查价格"""
        for record in self:
            if record.price < 0:
                raise ValidationError('价格不能为负数！')
    
    def _check_overlapping_pricing(self):
        """检查是否与现有定价重叠"""
        pricing_model = self.env['apollo.package.pricing']
        for record in self:
            domain = [
                ('package_id', '=', record.package_id.id),
                ('effective_date', '<=', record.expiry_date or '9999-12-31'),
            ]
            if record.expiry_date:
                domain.append(('expiry_date', '>=', record.effective_date))
            else:
                domain.append('|')
                domain.append(('expiry_date', '>=', record.effective_date))
                domain.append(('expiry_date', '=', False))
            
            overlapping = pricing_model.search(domain)
            if overlapping:
                overlapping_names = ', '.join([p.display_name for p in overlapping])
                raise ValidationError(f'套餐 {record.package_id.name} 在此时间段内已有定价：{overlapping_names}')
    
    def action_add_pricing(self):
        """添加定价"""
        if not self.package_id:
            raise UserError('未找到要添加定价的套餐！')
        
        # 检查重叠定价
        self._check_overlapping_pricing()
        
        # 创建定价记录
        pricing_vals = {
            'package_id': self.package_id.id,
            'price': self.price,
            'currency': self.currency,
            'effective_date': self.effective_date,
            'expiry_date': self.expiry_date,
        }
        
        new_pricing = self.env['apollo.package.pricing'].create(pricing_vals)
        
        # 显示成功通知并关闭向导
        return {
            'type': 'ir.actions.client',
            'tag': 'display_notification',
            'params': {
                'title': '套餐定价添加成功',
                'message': f'已成功为套餐 {self.package_id.name} 添加定价：{self.formatted_price}',
                'type': 'success',
                'sticky': False,
                'next': {
                    'type': 'ir.actions.act_window',
                    'name': '套餐定价',
                    'res_model': 'apollo.package.pricing',
                    'view_mode': 'form',
                    'res_id': new_pricing.id,
                    'target': 'current',
                }
            }
        }
    
    def action_add_and_close(self):
        """添加定价并关闭"""
        if not self.package_id:
            raise UserError('未找到要添加定价的套餐！')
        
        # 检查重叠定价
        self._check_overlapping_pricing()
        
        # 创建定价记录
        pricing_vals = {
            'package_id': self.package_id.id,
            'price': self.price,
            'currency': self.currency,
            'effective_date': self.effective_date,
            'expiry_date': self.expiry_date,
        }
        
        self.env['apollo.package.pricing'].create_pricing_with_log(
            pricing_vals, 
            reason='通过快速添加向导创建'
        )
        
        # 显示成功通知并关闭向导
        return {
            'type': 'ir.actions.act_window_close'
        }
